-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

-- this is a file containing definitions of asserts/matchers
-- for the "busted" testing framework

describe("DatatypeValidator class: ",
function()
	DatatypeValidator = require("datatype_validator")
	require("type_definitions")
	
	describe("get_type_of(): ",
	function()
		local datatype
		
		it("with strings",
		function()
			datatype = DatatypeValidator.get_type_of("string")

			assert.equals("string", datatype)
		end)
		
		it("with numbers",
		function()
			datatype = DatatypeValidator.get_type_of(100)

			assert.equals("number", datatype)
		end)

		it("with tables",
		function()
			datatype = DatatypeValidator.get_type_of({key = "value"})

			assert.equals("table", datatype)
		end)
	end)

	describe("cast_to(): ",
	function()
		local success, value
		
		it("from strings to numbers",
		function()
			value, success = DatatypeValidator.cast_to("10", "number")

			assert.equals(10, value)
			assert.equals(true, success)
		end)

		it("from numbers to string",
		function()
			value, success = DatatypeValidator.cast_to(10, "string")

			assert.equals("10", value)
			assert.equals(true, success)
		end)

		it("from numbers to string",
		function()
			value, success = DatatypeValidator.cast_to(10, "string")

			assert.equals("10", value)
			assert.equals(true, success)
		end)

		it("from strings to time",
		function()
			value, success = DatatypeValidator.cast_to("1h", "time")

			assert.equals(3600, value)
			assert.equals(true, success)
		end)

		it("impossible conversion",
		function()
			value, success = DatatypeValidator.cast_to({key = "value"}, "number")

			assert.equals(nil, value)
			assert.equals(false, success)
		end)
	end)

	-- testing this is hopeless because there are so many possible cases, this just test the most obvious ones
	describe("validate(): ",
	function()
		local output

		it("when given an empty ruleset",
		function()
			assert.errors(function() DatatypeValidator.validate({"arg1", "arg2", "arg3"}, {}) end)
		end)

		it("when given when given value rules",
		function()
			output = DatatypeValidator.validate({"value", 10}, 
			{
				{
					name = "arg1",
					type = "string",
					single = true
				},
				{
					name = "arg2",
					type = "number",
					single = true
				},
			})

			assert.same({arg1 = "value", arg2 = 10}, output)
		end)

		it("when given when given value rules but with too many arguments",
		function()
			assert.errors(function() 
				output = DatatypeValidator.validate({"value", 10, "extra argument"}, 
				{
					{
						name = "arg1",
						type = "string",
						single = true
					},
					{
						name = "arg2",
						type = "number",
						single = true
					},
				}
			)end)
		end)

		it("when given when given value rules that need to be converted",
		function()
			output = DatatypeValidator.validate({"value", "10"}, 
			{
				{
					name = "arg1",
					type = "string",
					single = true
				},
				{
					name = "arg2",
					type = "number",
					single = true
				},
			})

			assert.same({arg1 = "value", arg2 = 10}, output)
		end)

		it("when given when given value with many allowed types",
		function()
			output = DatatypeValidator.validate({"value", "10s"}, 
			{
				{
					name = "arg1",
					type = "string",
					single = true
				},
				{
					name = "arg2",
					type = {"time", "number"},
					single = true
				},
			})

			assert.same({arg1 = "value", arg2 = 10}, output)
		end)

		it("when given when given terminal rules",
		function()
			output = DatatypeValidator.validate({"terminal"}, 
			{
				{
					rule_type = "terminal",
					name = "name",
					value = "terminal"
				},
			})

			assert.same({name = "terminal"}, output)
		end)

		it("when given when given optional terminal rules",
		function()
			output = DatatypeValidator.validate({"value", "terminal", "value"}, 
			{
				{
					name = "arg1",
					type = "string",
					single = true
				},
				{
					rule_type = "terminal",
					name = "name",
					value = "terminal",
					optional = true
				},
				{
					name = "arg2",
					type = "string",
					single = true
				},
			})

			assert.same({name = "terminal", arg1 = "value", arg2 = "value"}, output)
		end)

		it("when given when given missing optional terminal rules",
		function()
			output = DatatypeValidator.validate({"value", "value"}, 
			{
				{
					name = "arg1",
					type = "string",
					single = true
				},
				{
					rule_type = "terminal",
					name = "name",
					value = "terminal",
					optional = true
				},
				{
					name = "arg2",
					type = "string",
					single = true
				},
			})

			assert.same({arg1 = "value", arg2 = "value"}, output)
		end)

		it("when given when given terminal rules that have multiple possible values",
		function()
			output = DatatypeValidator.validate({"term"}, 
			{
				{
					rule_type = "terminal",
					name = "name",
					value = {terminal = true, term = true},
				}
			})

			assert.same({name = "term"}, output)
		end)

		it("when given when given block rules",
		function()
			output = DatatypeValidator.validate({"arg1", "arg2", "arg3", 10}, 
			{
				{
					rule_type = "block",
					name = "name",
					type = "string"
				},
				{
					name = "number",
					type = "number",
					single = true
				}
			})

			assert.same({name = {"arg1", "arg2", "arg3"}, number = 10}, output)
		end)

		it("when given when given block rules with a terminator",
		function()
			output = DatatypeValidator.validate({"arg1", "arg2", "arg3", "end", "value"}, 
			{
				{
					rule_type = "block",
					name = "name",
					type = "string",
					terminator = 
					{
						value = "end"
					}
				},
				{
					name = "another_name",
					type = "string",
					single = true
				}
			})

			assert.same({"end", name = {"arg1", "arg2", "arg3"}, another_name = "value"}, output)
		end)
	end)

end)
