-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 


-- @fix this file requires the minetest namespace

local ShellContext = require("shell_context")
local Shell = require("language_constructs.shell")
local CommandInvocation = require("language_constructs.command_invocation")
local ParserContext = require("parser_context")
local DataTypeValidator = require("datatype_validator")
local helpers = require("helpers")

local Interpreter =
{
	instance_of = "Interpreter"
}

--- executes a shell
--	@string code the code to execute
--	@ShellContext initial_context the shell_context the shell will start with
--	@bool share_env whether the shell will share the environment of the passed ShellContext, this means that any parameter assignments will be persistent
--	@func complete_callback function called once execution ends
--	takes whether shell suceeded, error_message (when applicable) and shell context, parsing and evaluation time as arguments
function Interpreter.run_shell(code, initial_context, share_env, complete_callback)
	local t1 = minetest.get_us_time()
	if initial_context == nil then
		initial_context = {}
	end

	local shell_context = ShellContext:new(initial_context)
	shell_context.privilege_cache = minetest.get_player_privs(shell_context.env:get("name"))

	local parser_context = ParserContext:new({text = code})
	local shell = Shell:new()
	local success
	local message
	local parse_time
	local eval_time

	success, message = pcall(shell.parse, shell, parser_context, {}, {})

	if not success then
		complete_callback(false, message)
		return
	end

	parse_time = minetest.get_us_time() - t1
	t1 = minetest.get_us_time()

	success, message = pcall(shell.evaluate, shell, shell_context, share_env, function(shell_context)
		if complete_callback then
			complete_callback(shell_context.exit_status, shell_context.stderr, shell_context, parse_time, minetest.get_us_time() - t1)
		end
	end)
end

--- parses a shell, returns the parse tree that has the `evaluate(shell_context, share_environment)` method.
--	@string code the code to parse
--	@tab terminator_list list of keywords that will terminate the shell
--	@tab terminator_set set of *characters* that will terminate the shell
--	@treturn bool whether the parsing succeded
--	@treturn string error message, if any
--	@treturn tab the parse tree
--	@see Interpreter.run_shell
function Interpreter.parse_shell(code, terminator_list, terminator_set)

	local parser_context = ParserContext:new({text = code})
	local shell = Shell:new()
	local success
	local message

	success, message = pcall(shell.parse, shell, parser_context, {}, {})

	return success, message and string.format("parsing error: %s", message) or "parsing error: no error message", shell
end

--- similar to Interpreter.run_shell except it runs a single command instead of a shell
--	@string command
--	@ShellContext initial_context
--	@treturn bool success
--	@treturn string error_message
--	@treturn number parse_time time it took to parse the command
--	@treturn number eval_time time it took to evaluate the command
function Interpreter.run_command(command, initial_context)
	local t1 = minetest.get_us_time()
	if initial_context == nil then
		initial_context = {}
	end

	local shell_context = ShellContext:new(initial_context)
	shell_context.privilege_cache = minetest.get_player_privs(shell_context.env:get("name"))

	local parser_context = ParserContext:new({text = command})
	local command_invocation = CommandInvocation:new()
	local success
	local message
	local parse_time
	local eval_time

	success, message = pcall(command_invocation.parse, command_invocation, parser_context, helpers.white_chars_set)

	if not success then
		return false, message and string.format("parsing error: %s", message) or "parsing error: no error message", nil, nil
	end

	parse_time = minetest.get_us_time() - t1
	t1 = minetest.get_us_time()

	success, message = pcall(command_invocation.evaluate, command_invocation, shell_context)
	eval_time = minetest.get_us_time() - t1

	return success, message and string.format("evaluation error: %s", message) or "parsing error: no error message", 
		parse_time, eval_time
end

return Interpreter
