 -- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

--- @classmod LanguageConstruct_Expression
--	Describes a command invocation

local StringExpression
local NumericExpression
local InlineLua
local VectorExpression
local RangeExpression
local SelectorExpression
local TableExpression
local ProcessSubstitution
local ParameterExpansion


local helpers = require("helpers")

local Expression =
{
	instance_of = "LanguageConstruct_Expression",
}

Expression.__index = Expression

function Expression:new(t)
	if t == nil then
		t = {}
	end

	-- {{{ type checking
	if type(t) ~= "table" then error("t should be of type table. instead of " .. type(t)) end
	if t.expression and type(t.expression) ~= "table" then error("t.expression should be of type table. instead of " .. type(t.expression)) end
	-- }}}

	self = setmetatable(
	{
		expression = t.expression or nil
	}, Expression)

	return self
end

function Expression:parse(parser_context, terminator_set)
	-- {{{ type checking
	if parser_context.instance_of ~= "ParserContext" then error("parser_context should be an instance of ParserContext. instead of " .. parser_context.instance_of) end
	-- }}}

	StringExpression = StringExpression or require("language_constructs.string_expression")
	InlineLua = InlineLua or require("language_constructs.inline_lua")
	VectorExpression = VectorExpression or require("language_constructs.vector_expression")
	RangeExpression = RangeExpression or require("language_constructs.range_expression")
	SelectorExpression = SelectorExpression or require("language_constructs.selector_expression")
	NumericExpression = NumericExpression or require("language_constructs.numeric_expression")
	TableExpression = TableExpression or require("language_constructs.table_expression")
	ProcessSubstitution = ProcessSubstitution or require("language_constructs.process_substitution")
	ParameterExpansion = ParameterExpansion or require("language_constructs.parameter_expansion")

	-- lua pattern for any character belongning in the terminator set
	-- very hacky and temporary
	local terminator_set_pattern = "["
	for k, v in pairs(terminator_set) do
		if k == "]" then
			k = "%]"
		elseif k == ")" then
			k = "%)"
		end

		terminator_set_pattern = terminator_set_pattern .. k
	end
	terminator_set_pattern = terminator_set_pattern .. "]+"

	local char = parser_context:peek()

	-- @todo somehow figure out logic to predict whether the expression is a
	-- number, string, nodename or an itemname
	local expression
	if char == "`" then
		expression = InlineLua:new()
		expression:parse(parser_context)
	elseif char == "@" then
		expression = SelectorExpression:new()
		expression:parse(parser_context)
	elseif char == "[" then
		expression = TableExpression:new()
		expression:parse(parser_context)
	else
		-- for more complex types where i cant simply check the first character,
		-- use pattern matching
		-- this is a temporary thing untill i add a match function to all language constructs to do a better job

		-- vector
		if string.find(parser_context.text, [[^^?%([^)]*,[^)]*,[^)]*%)]], parser_context.character_index) then
			expression = VectorExpression:new()
			expression:parse(parser_context)

		-- range
		elseif string.find(parser_context.text, "^[<(][^)>]*:[^)>]*[)>]", parser_context.character_index) then
			expression = RangeExpression:new()
			expression:parse(parser_context)

		-- numeric expression
		elseif string.find(parser_context.text, "^-?%d+" .. terminator_set_pattern, parser_context.character_index)
			or string.find(parser_context.text, "^-?%d+%.%d+" .. terminator_set_pattern, parser_context.character_index) then
			expression = NumericExpression:new()
			expression:parse(parser_context, terminator_set)

		-- process substitution
		elseif string.find(parser_context.text, "^%$%(", parser_context.character_index) then
			expression = ProcessSubstitution:new()
			expression:parse(parser_context)

		-- parameter expansion
		elseif string.find(parser_context.text, "^%$", parser_context.character_index) then
			expression = ParameterExpansion:new()
			expression:parse(parser_context)

		-- strings
		else
			expression = StringExpression:new()
			expression:parse(parser_context, terminator_set)
		end
	end

	for key, value in pairs(expression) do
		self[key] = value
	end

	setmetatable(self, getmetatable(expression))
end

return Expression
