-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

--- @classmod LanguageConstruct_ParameterExpansion
--	Describes a parameter expansion
local helpers = require("helpers")
local DatatypeValidator
local StringExpression

-- @todo once minetest uses new lua interpreter, change back to having the modifier be a bitmap

--- @fixme somehow make Modifiers not show up as a member of ParameterExpansion in the documentation

--- @table Modifiers
--	the documentation is finicky, this isnt actually a member of the class
--
--	@field none does nothing before evaluating <br>
--	format: ${var} or $var
--
--	@field fallback if parameter with given name *var* is undefined, evaluates to *word* <br>
--	format: ${var:-word}
--
-- 	@field fallback_set if parameter with given name *var* is undefined, while evaluating sets the parameter to *word* and returns it <br>
-- 	format: ${var:=word}
--
-- 	@field guard if parameter with given name *var* is undefined, raises an error with message *word*  <br>
-- 	format: ${var:?word}
--
-- 	@field offset evaulates to a substring of the parameter's value that starts at *offset* and goes to the end of the string <br>
-- 	format: ${var:offset}
--
--	@field substring evaluates to a substring of the parameter's value that starts as *offset* and has a length of *length* <br>
--	format: ${var:offset:length}
--
--
--	@field substitution evaluates to the parameter's value that has all matching *patterns* substituted with *string*, uses lua's string.gsub() syntax <br> format: ${var/pattern/string}
--
--	@see string.gsub
local Modifiers =
{
	["none"] = 0,

	["fallback"] = 1,

	["fallback_set"] = 3,

	["guard"] = 4,

	["offset"] = 5,

	["substring"] = 6,

	["substitution"] = 7
}

--- @table AdditionalModifiers
--	these are additional modifiers that can be combined with the ones previously mentioned, but not with eachother
--
--	@field length evaluates to the length of the parameter's value after applying the modifiers
-- 	format: ${#var}
--
--	@field indirection evaluates to the value of a parameter whose name is the value of the passed parameter (with previous mondifiers applied)
--	example: <br>
--	<code>
--	foo="first value"; <br>
--	foo_pointer="foo"; <br>
--	echo ${!foo_pointer} #prints "first value" <br>
--	</code>
--	format: ${!var}
local AdditionalModifiers =
{
	["none"] = 0,
	["length"] = 1,
	["indirection"] = 2,
}

local ParameterExpansion =
{
	instance_of = "LanguageConstruct_ParameterExpansion",
	valid_modifiers = Modifiers, -- exposing the valid Modifiers table for testing
	valid_additional_modifiers = AdditionalModifiers, -- exposing the valid AdditionalModifiers table for testing
}

ParameterExpansion.__index = ParameterExpansion

function ParameterExpansion:new(t)
	if t == nil then
		t = {}
	end

	-- {{{ type checking
	if type(t) ~= "table" then error("t should be of type table. instead of " .. type(t)) end
	if t.parameter_name and type(t.parameter_name) ~= "string" then error("t.parameter_name should be of type string. instead of " .. type(t.parameter_name)) end
	if t.modifier and type(t.modifier) ~= "number" then error("t.modifier should be of type number. instead of " .. type(t.modifier)) end
	if t.modifier_args and type(t.modifier_args) ~= "table" then error("t.modifier_ar should be of type table. instead of " .. type(t.modifier_ar)) end



	-- }}}

	self = setmetatable(
	{

		--- @string[opt=""] parameter_name name of the parameter to fetch
		parameter_name = t.parameter_name or "",

		--- @number[opt=0] modifier bitmask signifying what needs to be done before returning the parameter's value <br>
		--	only length and indirection can be OR'ed with other modifiers
		--	@see Modifiers
		modifier = t.modifier or Modifiers.none, -- enum type from Modifiers

		additional_modifier = t.modifier or AdditionalModifiers.none, -- enum type from AdditionalModifiers

		--- @tab[opt={}] modifier_args list of arguments for the modifier while evaluating
		modifier_args = t.modifier_args or {}
	}, ParameterExpansion)

	return self
end

--- evaluates the parameter expansion and returns the value
--	@ShellContext context
--	@treturn string the value of the parameter
--
-- {{{ evaluating
function ParameterExpansion:evaluate(context)
	-- {{{ type checking
		if type(self.parameter_name) ~= "string" then error("self.parameter_name should be of type string. instead of " .. type(self.parameter_name)) end
		if type(self.modifier) ~= "number" then error("self.modifier should be of type number. instead of " .. type(self.modifier)) end
		if type(self.additional_modifier) ~= "number" then error("self.additional_modifier should be of type number. instead of " .. type(self.additional_modifier)) end
	-- }}}

	DatatypeValidator = DatatypeValidator or require("datatype_validator")

	local parameter_value = context.env:get(self.parameter_name)
	local parameter_value_type = DatatypeValidator.get_type_of(parameter_value)

	if not (self.modifier == Modifiers.none and self.additional_modifier == Modifiers.none)
		and parameter_value_type ~= "string" and parameter_value ~= nil then
		error("cant use modifiers on non string parameters")
	end

	-- applying modifiers
	if self.modifier == Modifiers.fallback then
		if not parameter_value or #parameter_value == 0 then
			parameter_value = self.modifier_args[1]:evaluate(context)
		end

	elseif self.modifier == Modifiers.fallback_set then
		if not parameter_value or #parameter_value == 0 then
			parameter_value = self.modifier_args[1]:evaluate(context)
			context.env:set(self.parameter_name, parameter_value)
		end

	elseif self.modifier == Modifiers.guard then
		if not parameter_value or #parameter_value == 0 then
			context:error(self.modifier_args[1]:evaluate(context))
		end

	elseif self.modifier == Modifiers.offset then
		if parameter_value then
			local offset = tonumber(self.modifier_args[1]:evaluate(context))
			if type(offset) == "number" then
				parameter_value = string.sub(parameter_value, offset)
			else
				error("failed to convert the offset value to a number")
			end
		end

	elseif self.modifier == Modifiers.substring then
		if parameter_value then
			local offset = tonumber(self.modifier_args[1]:evaluate(context))
			local length = tonumber(self.modifier_args[2]:evaluate(context))
			if type(offset) == "number" and type(length) == "number" then
				parameter_value = string.sub(parameter_value, offset, length)
			else
				error("failed to convert the offset or length to a number")
			end
		end

	elseif self.modifier == Modifiers.substitution then
		if parameter_value then
			parameter_value = string.gsub(parameter_value, self.modifier_args[1]:evaluate(context), self.modifier_args[2]:evaluate(context))
		end
	end

	if self.additional_modifier == AdditionalModifiers.indirection then
		parameter_value = context.env:get(parameter_value)
	end

	if self.additional_modifier == AdditionalModifiers.length then
		if parameter_value then
			parameter_value = #parameter_value
		else
			parameter_value = 0
		end
	end

	if parameter_value == nil then
		parameter_value = ""
	end

	return parameter_value
end
-- }}}

--- parses a parameter expansion
--	@ParserContext parser_context
--	@treturn nil
--
-- {{{ parsing
function ParameterExpansion:parse(parser_context)
	StringExpression = StringExpression or require("language_constructs.string_expression")

	local char
	local parameter_name = ""
	local modifier_arg
	char = parser_context:consume(2)

	if helpers.is_valid_identifier_char(char, true) then -- expected: $<var>
		self.modifier = self.valid_modifiers.none
		parameter_name = parameter_name .. char

		-- state: $<first character>*

		while helpers.is_valid_identifier_char(parser_context:peek()) do
			parameter_name = parameter_name .. parser_context:consume()
		end

		-- state: $<var>*

	elseif char == "{" then -- expected: ${<var>}
		char = parser_context:consume()

		-- state: ${*

		if char == "#" then
			self.additional_modifier = AdditionalModifiers.length
			char = parser_context:consume()
		elseif char == "!" then
			self.additional_modifier = AdditionalModifiers.indirection
			char = parser_context:consume()
		elseif not helpers.is_valid_identifier_char(char, true) then
			error("invalid parameter name")
		end


		parameter_name = parameter_name .. char

		while helpers.is_valid_identifier_char(parser_context:peek()) do
			parameter_name = parameter_name .. parser_context:consume()
		end

		-- state: ${<var>*

		char = parser_context:consume()

		if char == ":" then
			-- state: ${<var>:*

			char = parser_context:consume()

			-- state: ${<var>:*

			if char == "-" then
				-- state: ${<var>:-*

				self.modifier = self.valid_modifiers.fallback

				modifier_arg = StringExpression:new()
				modifier_arg:parse(parser_context,
				{ -- terminator set
					["}"] = true
				})

				table.insert(self.modifier_args, modifier_arg)

				--state: ${<var>:-<word>*

				parser_context:advance()

				--state: ${<var>:-<word>}*
			elseif char == "=" then
				-- state: ${<var>:=*

				self.modifier = self.valid_modifiers.fallback_set

				modifier_arg = StringExpression:new()
				modifier_arg:parse(parser_context,
				{ -- terminator set
					["}"] = true
				})

				table.insert(self.modifier_args, modifier_arg)

				--state: ${<var>:=<word>*

				parser_context:advance()

				--state: ${<var>:=<word>}*
			elseif char ==  "?" then
				-- state: ${<var>:?*

				self.modifier = self.valid_modifiers.guard

				modifier_arg = StringExpression:new()
				modifier_arg:parse(parser_context,
				{ -- terminator set
					["}"] = true
				})

				table.insert(self.modifier_args, modifier_arg)

				--state: ${<var>:?<word>*

				parser_context:advance()

				--state: ${<var>:?<word>}*
			else
				-- state: ${<var>:<number>*

				parser_context:rewind()

				-- state: ${<var>:*<number>

				modifier_arg = StringExpression:new()
				modifier_arg:parse(parser_context,
				{ -- terminator set
					["}"] = true,
					[":"] = true
				})

				table.insert(self.modifier_args, modifier_arg)

				char = parser_context:consume()


				if char == "}" then
					-- state: ${<var>:<offset>}*
					self.modifier = self.valid_modifiers.offset

				elseif char == ":" then
					-- state: ${<var>:<offset>:*
					self.modifier = self.valid_modifiers.substring

					modifier_arg = StringExpression:new()
					modifier_arg:parse(parser_context,
					{ -- terminator set
						["}"] = true
					})

					table.insert(self.modifier_args, modifier_arg)

					parser_context:advance()

					--state: ${var:<offset>:<length>}*
				else
					error("unexpected character")
				end

				--state: ${var:<numbers>}* or
				--state: ${var:<numbers>:<length>}* or
			end


		elseif char == "/" then
			-- state: ${<var>/*
			self.modifier = Modifiers.substitution

			modifier_arg = StringExpression:new()
			modifier_arg:parse(parser_context,
			{ -- terminator set
				["}"] = true,
				["/"] = true,
			})

			table.insert(self.modifier_args, modifier_arg)

			-- state: ${<var>/<pattern>*

			char = parser_context:consume()

			-- state: ${<var>/<pattern>/*

			if char == "/" then
				modifier_arg = StringExpression:new()
				modifier_arg:parse(parser_context,
				{ -- terminator set
					["}"] = true,
					["/"] = true,
				})

				table.insert(self.modifier_args, modifier_arg)

				-- state: ${<var>/<pattern>/<substitution>*

				parser_context:advance()

				-- state: ${<var>/<pattern>/<substitution>}*
			else
				error("unexpected character")
			end

		end

		-- state: ${var}*
	else
		error()
	end

	self.parameter_name = parameter_name
end
-- }}}

return ParameterExpansion
