-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

local helpers = require("helpers")
local StringExpression = require("language_constructs.string_expression")
local Expression = require("language_constructs.expression")
local Shell = require("language_constructs.shell")
local DatatypeValidator = require("datatype_validator")

local IfStatement =
{
	instance_of = "LanguageConstruct_IfStatement",
}

IfStatement.__index = IfStatement

function IfStatement:new(t)
	t = t or {}
	self = setmetatable(
	{
		expression = t.expression,
		success_subshell = t.success_subshell,
		elseif_expressions = t.elseif_expressions or {},
		elseif_subshells = t.elseif_subshells or {},
		fail_subshell = t.fail_subshell,
	}, IfStatement)

	return self
end

local function evaluate_as_bool(value)
	if type(value) == "table" and value.instance_of == "TypedList" then
		return #value.container ~= 0
	end

	local datatype = DatatypeValidator.get_type_of(value)

	if type(value) == "table" and value.instance_of == "TypedList" then
		return #value > 0
	elseif DatatypeValidator.registered_types[datatype]
		and DatatypeValidator.registered_types[datatype].to_bool then
		return DatatypeValidator.registered_types[datatype].to_bool(value)
	else
		return false
	end
end

function IfStatement:evaluate(context)
	-- {{{ type checking
	if context.instance_of ~= "ShellContext" then error("context should be an instance of ShellContext. instead of " .. context.instance_of) end
	-- }}}


	local evaluated_expression = self.expression:evaluate(context)
	local success = evaluate_as_bool(evaluated_expression)

	if success then
		self.success_subshell:evaluate(context, true)
	else
		for i = 1, #self.elseif_subshells do
			success = evaluate_as_bool(self.elseif_expressions[i]:evaluate(context))

			if success then
				self.elseif_subshells[i]:evaluate(context, true)
				break
			end
		end

		if not success and self.fail_subshell then
			self.fail_subshell:evaluate(context, true)
		end
	end

	if context.exit_status ~= 0 then
		error("if statement: " .. context.stderr)
	end
end

function IfStatement:parse(parser_context)
	-- {{{ type checking
	if parser_context.instance_of ~= "ParserContext" then error("parser_context should be an instance of ParserContext. instead of " .. parser_context.instance_of) end
	-- }}}
	local success
	
	-- state: *
	success = parser_context:match_fast("if", true)

	if not success then
		error("expected 'if'")
	end

	parser_context:skip_until(helpers.white_chars_set, true)

	-- state: if*

	self.expression = Expression:new()
	self.expression:parse(parser_context, helpers.white_chars_set)

	parser_context:skip_until(helpers.white_chars_set, true)

	-- state: if <expression>*
	
	success = parser_context:match_fast("then", true)

	if not success then
		error("expected 'then'")
	end

	-- state: if <expression> then*

	parser_context:skip_until(helpers.white_chars_set, true)

	self.success_subshell = Shell:new()
	self.success_subshell:parse(parser_context, {"end", "else", "elseif"})

	-- state: if <expression> then
	-- <subshell>
	-- *

	local success, match = parser_context:match({"end", "else", "elseif"}, true)

	if not success then
		error("expected any of 'end', 'else', 'elseif'")
	end

	if match == "else" then
		-- state: if <expression> then
		-- <subshell>
		-- else
		-- *

		parser_context:skip_until(helpers.white_chars_set, true)
		self.fail_subshell = Shell:new()
		self.fail_subshell:parse(parser_context, {"end"})

		-- state: if <expression> then
		-- <subshell>
		-- else
		-- <subshell>
		-- *

		parser_context:advance(3)
	elseif match == "elseif" then
		-- the state comments are becoming too verbose, im just going without them for this case
		local i = 1
		while match == "elseif" do
			parser_context:skip_until(helpers.white_chars_set, true)

			self.elseif_expressions[i] = Expression:new()
			self.elseif_expressions[i]:parse(parser_context, helpers.white_chars_set)
			parser_context:skip_until(helpers.white_chars_set, true)

			success = parser_context:match_fast("then", true)

			if not success then
				error("expected 'then'")
			end

			parser_context:skip_until(helpers.white_chars_set, true)
			self.elseif_subshells[i] = Shell:new()
			self.elseif_subshells[i]:parse(parser_context, {"end", "else", "elseif"})

			success, match = parser_context:match({"end", "else", "elseif"}, true)

			if not success then
				error("expected any of 'end', 'else', 'elseif'")
			end

			i = i + 1
		end

		if match == "else" then
			parser_context:skip_until(helpers.white_chars_set, true)
			self.fail_subshell = Shell:new()
			self.fail_subshell:parse(parser_context, {"end"})
		end
	elseif match ~= "end" then
		error("unexplainable parser horrors")
	end

	-- state: if <expression> then
	-- <subshell>
	-- end*
end

return IfStatement
