-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

-- this file implements commands for messaging other players
-- and the ability for moderators to see whats being said

local storage = minetest.get_mod_storage()

-- set of player names who should intercept all private messages
local spies = {}

-- dictionary:
-- keys: string player name
-- value: a set of player names of blocked players
local player_blocks = minetest.deserialize(storage:get_string("player_blocks") or "") or {}

local reviewers = {}

local recent_private_messages = {}

local use_chat_filter = false
if minetest.settings:get_bool("lush_filter") then
	if minetest.get_modpath("filterpp_lib") then
		use_chat_filter = true

		minetest.register_privilege("filter_reviewer",
		{
			description = "players with this privilege will get reports about swears whenever they happen",
			give_to_admin = true,

			on_grant = function(name, granter_name)
				table.insert(reviewers, name)
			end,

			on_revoke = function(name, granter_name)
				local idx = table.indexof(reviewers, name)

				table.remove(reviewers, idx)
			end
		})

		minetest.register_on_joinplayer(function(player)
			if minetest.check_player_privs(player:get_player_name(), "filter++_reviewer") then
				table.insert(reviewers, player:get_player_name())
			end
		end)

		filterpp_lib.register_on_violation(function(name, message, bad_words)
			local report_message = string.format("%s said illegal words: ", name)
			local _, whole_bad_words = filterpp_lib.filter_text(message, true)

			for i = 1, #whole_bad_words do
				report_message = report_message .. string.format('"%s" ', message:sub(unpack(whole_bad_words[i])))
			end

			for _, reviewer_name in pairs(reviewers) do
				minetest.chat_send_player(reviewer_name, report_message)
			end

			if minetest.get_modpath("warn") then
				warn_system.give_warn(name, report_message)
			end
		end)
	else
		minetest.log("error", "filter++ lib needs to be installed to use chat filtering\n if you dont want to filter then you can disable it in mod settings (lush_filter)")
	end
end

--- register privileges

minetest.register_privilege("socialspy",
{
	description = "whether the player can spy on other's private messages",
	give_to_admin = true
})

minetest.register_privilege("socialspy_protected",
{
	description = "private messages of player with this priviledge cant be spied on",
	give_to_admin = true
})

minetest.register_privilege("immutable",
{
	description = "players with this priviledge cannot be blocked",
	give_to_admin = true
})


local lush_override_commands = minetest.settings:get_bool("lush_override_commands")

-- setting default value
if lush_override_commands == nil then
	lush_override_commands = true
end

if lush_override_commands then
	minetest.unregister_chatcommand("me")
end

--- sends the message to every spy
--	@string from the name of the player who sent the message
--	@string to the name of the player who will be recieving the message
--	@string message the message that was sent
--	@treturn nil
local function notify_spies(from, to, message)
	if minetest.check_player_privs(from, "socialspy_protected") or minetest.check_player_privs(to, "socialspy_protected") then
		return
	end

	for name, _ in pairs(spies) do
		-- dont send duplicate messages to the spy if he is the recipient
		if not (name == from or name == to) then
			minetest.chat_send_player(name, string.format("SPY: %s -> %s: %s", from, to, message))
		end
	end
end

lush.Command:new(
{
	name = "bc",
	description = "sends a server broadcast",
	params = "<message>",
	privs = {server = true},
	arguments = 
	{
		{
			-- i dont want to force the user to double quote every message
			-- if their message has spaces, so i capture every argument afterwards
			-- and table.concat() it
			rule_type = "block",
			name = "messages",
			type = "string",
			required = true
		}
	},
	callback = function(context, args, options)
		local message = table.concat(args.messages, " ")
		
		minetest.chat_send_all(string.format("[SERVER] %s", message))
	end,
})

lush.Command:new(
{
	name = "msg",
	description = "sends a private message",
	params = "<player> <message>",
	privs = {shout = true},
	arguments = 
	{
		{
			name = "recipient",
			type = "player_name",
			single = true,
		},
		{
			-- i dont want to force the user to double quote every message
			-- if their message has spaces, so i capture every argument afterwards
			-- and table.concat() it
			rule_type = "block",
			name = "messages",
			type = "string",
			required = true
		}
	},
	callback = function(context, args, options)
		if args.recipient == context.env:get("name") then
			error("you cant message yourself")
		end

		-- when recipient blocked the player trying to send a private message
		if player_blocks[args.recipient][context.env:get("name")] then
			error(string.format("%s blocked you", args.recipient))
		end

		local message = table.concat(args.messages, " ")
		
		notify_spies(context.env:get("name"), args.recipient, message)
		recent_private_messages[context.env:get("name")] = args.recipient

		minetest.chat_send_player(args.recipient, string.format("%s messaged you: %s", context.env:get("name"), message))
	end,
})

lush.Command:new(
{
	name = "block",
	description = "blocks a player so he cant message you",
	arguments = 
	{
		{
			name = "player",
			type = "player_name",
			single = true,
		},
	},
	callback = function(context, args, options)
		if minetest.check_player_privs(args.player, "immutable") then
			error(string.format("you cant block %s", args.player))
		end

		player_blocks[context.env:get("name")][args.player] = true
		storage:set_string("player_blocks", minetest.serialize(player_blocks))
	end,
})

lush.Command:new(
{
	name = "unblock",
	description = "unblocks a player",
	arguments = 
	{
		{
			name = "player",
			type = "player_name",
			single = true,
		},
	},
	callback = function(context, args, options)
		player_blocks[context.env:get("name")][args.player] = nil
		storage:set_string("player_blocks", minetest.serialize(player_blocks))
	end,
	description = "unblocks a player",
})

lush.Command:new(
{
	name = "lsblocks",
	description = "lists all players you've blocked",
	arguments = {},
	callback = function(context, args, options)
		for name, _ in pairs(player_blocks[context.env:get("name")]) do
			context.stdout:push(name)
		end
	end,
})

minetest.register_on_joinplayer(function(player, last_login)
	player_blocks[player:get_player_name()] = {}
	storage:set_string("player_blocks", minetest.serialize(player_blocks))
end)

minetest.register_on_chat_message(function(name, message)
	local char = string.sub(message, 1, 1)

	if char == "!" or char == "/" then
		return false
	end


	if not minetest.check_player_privs(name, "shout") then
		minetest.chat_send_player(name, "you cant speak")
		return false
	end

	if use_chat_filter then
		local should_censor, bad_words = filterpp_lib.filter_text(message)

		if should_censor then
			filterpp_lib.report_violation(name, message, bad_words)

			local censored_message = ""
			local last_badword_idx = 1

			for _, wordinfo in pairs(bad_words) do
				if wordinfo[1] ~= 1 then
					censored_message = censored_message .. message:sub(last_badword_idx, wordinfo[1] - 1)
				end
				censored_message = censored_message .. string.rep("*", wordinfo[2] - wordinfo[1] + 1)
				last_badword_idx = wordinfo[2] + 1
			end

			message = censored_message .. message:sub(last_badword_idx)
		end
	end

	if char == "@" then
		local _, end_index, recipient_name = message:find("^(%S+)%s+", 2)

		if not recipient_name then
			if recent_private_messages[name] then
				recipient_name = recent_private_messages[name]
			else
				minetest.chat_send_player(name, "no recent private messages")
				return true
			end

		end

		if player_blocks[recipient_name][name] then
			minetest.chat_send_player(name, string.format("%s blocked you", recipient_name))
			return true
		end
			
		minetest.chat_send_player(recipient_name, minetest.format_chat_message(name, minetest.colorize("cyan", "[PM] ") .. message:sub(end_index or 3, -1):trim()))
		recent_private_messages[name] = recipient_name
	else
		local recipient_name
		for _, recipient in pairs(minetest.get_connected_players()) do
			recipient_name = recipient:get_player_name()
			if not player_blocks[recipient_name][name] then
				minetest.chat_send_player(recipient_name, minetest.format_chat_message(recipient_name, message))
			end
		end
	end

	return true
end)

