-- Copyright (C) 2024 rstcxk
--
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
--
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.

local lush_override_commands = core.settings:get_bool("lush_override_commands")

-- setting default value
if lush_override_commands == nil then
	lush_override_commands = true
end

if lush_override_commands then
	core.unregister_chatcommand("mods")
end

core.register_privilege("lush_heal",
{
	description = "whether the player can use the heal command",
	give_to_singleplayer = false
})

core.register_privilege("lush_damage",
{
	description = "whether the player can use the damage/kill command",
	give_to_singleplayer = false
})

core.register_privilege("invisible",
{
	description = "whether the player can use the invis command",
	give_to_singleplayer = false
})

core.register_privilege("clear",
{
	description = "whether the player can use the clear command",
	give_to_singleplayer = false
})

core.register_privilege("fix",
{
	description = "whether the player can use the fix_items command",
	give_to_singleplayer = false
})

local max = tonumber(core.settings:get("lush_max_lag")) or 2.0

local lag = 0
local last = core.get_us_time()
core.register_globalstep(function()
		if lag <= 0 then
			return
		end

		local exp = last + lag * 1000000
		while core.get_us_time() < exp do
			-- looping until time passes
		end

		last = exp
	end)

lush.Command:new(
{
	name = "lag",
	description = "lags out the server so each global step takes a specific amount of seconds",
	privs = {server = true},
	arguments =
	{
		{
			name = "time_per_globalstep",
			type = "time",
			single = true
		}
	},
	callback = function(ctx, args, options)
		if args.time_per_globalstep > max then
			error("time per globalstep can not exceed " .. tostring(max) .. " seconds")
		end

		lag = args.time_per_globalstep
	end,
})

lush.Command:new(
{
	name = "settings",
	description = "sets server configuration settings",
	privs = {server = true},
	arguments =
	{
		{
			rule_type = "terminal",
			value = {set = true}
		},
		{
			name = "name",
			type = "string",
			single = true
		},
		{
			name = "value",
			type = "string",
			single = true
		}
	},
	callback = function(ctx, args, options)
		core.settings:set(args.name, args.value)
	end,
})

lush.Command:new(
{
	name = "days",
	description = "prints the amounts of days passed",
	privs = {debug = true},
	arguments = {},
	callback = function(ctx, args, options)
		ctx.stdout:push(core.get_day_count())
	end,
})

lush.Command:new(
{
	name = "now",
	description = "returns the UNIX time",
	arguments = {},
	callback = function(ctx, args, options)
		ctx.stdout:push(os.time())
	end,
})

local storage = core.get_mod_storage()

local time_update_job

local function time_update_func(time)
	core.set_timeofday(time)
	time_update_job = core.after(1, time_update_func, time)
end

if storage:contains("lush_permament_time") and storage:get_float("lush_permament_time") ~= -1 then
	time_update_job = core.after(1, time_update_func, storage:get_float("lush_permament_time"))
end

lush.Command:new(
{
	name = "time",
	description = "sets time",
	options = {["permament"] = false},
	privs = {settime = true},
	arguments =
	{
		{
			name = "time",
			type = {"time", "string"},
			single = true
		}
	},
	callback = function(ctx, args, options)
		local time_of_day = 0

		if type(args.time) == "string" then
			if args.time == "midnight" then
				time_of_day = 0
			elseif args.time == "night" then
				time_of_day = 0.8
			elseif args.time == "sunset" then
				time_of_day = 0.75
			elseif args.time == "noon" then
				time_of_day = 0.5
			elseif args.time == "day" then
				time_of_day = 0.3
			elseif args.time == "sunrise" then
				time_of_day = 0.2
			else
				error(string.format("%s isnt a recgonized time", args.time))
			end
		else
			if args.time >= 86400 or args.time < 0 then
				error("time must be between 0 seconds and 24 hours")
			end

			time_of_day = args.time / 86400 -- seconds in a day
		end

		core.set_timeofday(time_of_day)

		if options["permament"] then
			if time_update_job then
				time_update_job:cancel()
			end
			time_update_job = core.after(1, time_update_func, time_of_day)
			storage:set_float("lush_permament_time", time_of_day)
		else
			if time_update_job then
				time_update_job:cancel()
			end
			storage:set_float("lush_permament_time", -1)
		end
	end,
})

local pickers = {}

lush.Command:new(
{
	name = "pick",
	description = "punch a node to print its name/position",
	arguments =
	{
		{
			rule_type = "terminal",
			name = "mode",
			value = {pos = true, name = true}
		}
	},
	callback = function(ctx, args, options)
		if pickers[ctx.env:get("name")] then
			error("can't pick a node because you are picking one already")
		end


		ctx:interrupt()
		pickers[ctx.env:get("name")] = {ctx = ctx, mode = args.mode}
	end,
})

core.register_on_punchnode(function(pos, node, puncher)
	local name = puncher:get_player_name()
	if pickers[name] then
		if pickers[name].mode == "pos" then
			pickers[name].ctx.stdout:push(pos)
		else
			pickers[name].ctx.stdout:push(node.name)
		end
		pickers[name].ctx:stop_interruption()
		pickers[name] = nil
	end
end
)

lush.Command:new(
{
	name = "launch",
	description = "adds velocity to entities",
	privs = {lush_damage = true},
	arguments =
	{
		{
			name = "objects",
			type = "objectref",
			single = false
		},
		{
			name = "vector",
			type = "vector",
			single = true
		}
	},
	callback = function(ctx, args, options)
		for _, v in args.objects:iterator() do
			v:add_velocity(args.vector)
		end
	end,
})

lush.Command:new(
{
	name = "teleport",
	description = "teleports entities to position",
	options = {},
	privs = {teleport = true},
	arguments =
	{
		{
			name = "objects",
			type = "objectref",
		},
		{
			rule_type = "terminal",
			value = "to",
			optional = true,
		},
		{
			name = "pos",
			type = "vector",
			single = true
		}
	},
	callback = function(ctx, args, options)
		local lm = 31007 -- equals MAX_MAP_GENERATION_LIMIT in C++
		local pos = args.pos
		if pos.x < -lm or pos.x > lm or pos.y < -lm or pos.y > lm
				or pos.z < -lm or pos.z > lm then
			error("position out of map's bounds")
		end

		if (ctx.env:get("name") ~= args.objects:get(1):get_player_name() or #args.objects > 1) and not ctx.privilege_cache["bring"] then
			error("you need the 'bring' privilege to teleport anything other than yourself")
		end

		for _, v in args.objects:iterator() do
			if v:get_pos() and not v:get_attach() then
				v:set_pos(args.pos)
			end
		end

	end,
})

lush.Command:new(
{
	name = "kill",
	description = "kills entities",
	privs = {lush_damage = true},
	arguments =
	{
		{
			name = "objects",
			type = "objectref",
		},
	},
	callback = function(ctx, args, options)
		for i, v in args.objects:iterator() do
			lush.helpers.deal_damage(v, v:get_hp(), nil, true)
		end

	end,
})

lush.Command:new(
{
	name = "damage",
	description = "deals damage to entities",
	options =
	{
		["type"] = true,
	},
	privs = {lush_damage = true},
	arguments =
	{
		{
			name = "targets",
			type = "objectref",
			single = false,
		},
		{
			name = "amount",
			type = "number",
			single = true,
		},
	},
	callback = function(ctx, args, options)
		local damage_type = "set_hp" or options["type"]

		if type(damage_type) ~= "string" then
			error("damage type should be a string")
		end

		for _, v in args.targets:iterator() do
			lush.helpers.deal_damage(v, args.amount, nil, true)
		end

	end,
})

lush.Command:new(
{
	name = "spawnpoint",
	description = "sets the respawn point for players",
	privs = {server = true},
	arguments =
	{
		{
			name = "names",
			type = "player_name",
			single = false,
		},
		{
			name = "pos",
			type = "vector",
			single = true,
		},
	},
	callback = function(ctx, args, options)
		for _, v in args.names:iterator() do
			lush.spawnpoints[v] = args.pos
		end

		lush.Storage.save("spawnpoints", lush.spawnpoints)
	end,
})

lush.Command:new(
{
	name = "clear_spawnpoint",
	description = "removes the previously set spawnpoint",
	privs = {server = true},
	arguments =
	{
		{
			name = "names",
			type = "player_name",
			single = false,
		},
	},
	callback = function(ctx, args, options)
		for _, v in args.names:iterator() do
			lush.spawnpoints[v] = nil
		end

		lush.Storage.save("spawnpoints", lush.spawnpoints)
	end,
})

lush.Command:new(
{
	name = "heal",
	description = "heals entities by an amount",
	privs = {lush_heal = true},
	arguments =
	{
		{
			name = "targets",
			type = "objectref",
			single = false,
		},
		{
			name = "amount",
			type = "number",
			single = true,
		},
	},
	callback = function(ctx, args, options)
		for _, v in args.targets:iterator() do
			lush.helpers.deal_damage(v, -args.amount, nil, false)
		end
	end,
})

lush.Command:new(
{
	name = "list",
	description = "lists online players",
	privs = {debug = true},
	arguments = {},
	callback = function(ctx, args, options)
		for _, player in ipairs(core.get_connected_players()) do
			ctx.stdout:push(player:get_player_name())
		end
	end,
})

lush.Command:new(
{
	name = "seed",
	description = "prints the world's seed",
	privs = {debug = true},
	arguments = {},
	callback = function(ctx, args, options)
		ctx.stdout:push(core.get_mapgen_setting("seed"))
	end,
})

lush.Command:new(
{
	name = "lsmods",
	description = "prints active mods to STDOUT",
	privs = {debug = true},
	arguments = {},
	callback = function(ctx, args, options)
		for _, v in ipairs(core.get_modnames()) do
			ctx.stdout:push(v)
		end
	end,
})

-- credit to SkyBuilder1717's Essentials mod
-- from whom this implementation is partly ripped
lush.Command:new(
{
	name = "invis",
	description = "makes entities invisible",
	privs = {invisible = true},
	arguments =
	{
		{
			type = "player",
			name = "players",
			single = false
		},
		{
			type = "string",
			name = "is_enabled",
			single = true
		}
	},
	callback = function(ctx, args, options)
		local prop

		if args.is_enabled == "on" then
			prop = {
				visual_size = {x = 0, y = 0, z = 0},
				is_visible = false,
				nametag_color = {r=255,g=255,b=255,a=255},
				pointable = false,
				makes_footstep_sound = false,
				show_on_minimap = false,
			}
		elseif args.is_enabled == "off" then
			prop = {
				visual_size = {x = 1, y = 1, z = 1},
				is_visible = true,
				nametag_color = {r=255,g=255,b=255,a=0},
				pointable = true,
				makes_footstep_sound = true,
				show_on_minimap = true,
			}
		else
			error("invalid argument: '" .. tostring(args.is_enabled) .. "', should be either on or off")
		end

		for _, v in args.players:iterator() do
			v:set_properties(prop)

			if args.is_enabled == "on" then
				v:get_meta():set_int("invisible", 1)
			elseif args.is_enabled == "off" then
				v:get_meta():set_int("invisible", 0)
			end
		end
	end,
})

lush.Command:new(
{
	name = "give",
	description = "gives items",
	options =
	{
		["wear"] = true,
		["amount"] = true,
		["meta"] = true
	},
	privs = {give = true},
	arguments =
	{
		{
			type = "player",
			name = "players",
			single = false
		},
		{
			type = {"node_name", "string"},
			name = "item_name",
			single = true
		},
	},
	callback = function(ctx, args, options)
		local stack = ItemStack(args.item_name)

		if options.meta then
			stack:get_meta():from_table({fields = options.meta})
		end

		if options.wear then
			stack:set_wear(options.wear)
		end

		if options.amount then
			stack:set_count(options.amount)
		end

		local inv
		for _, object in args.players:iterator() do
			inv = object:get_inventory()

			inv:add_item("main", stack)
		end

	end,
})

lush.Command:new(
{
	name = "summon",
	description = "summons entities",
	options =
	{
		["count"] = true,
	},
	privs = {give = true},
	arguments =
	{
		{
			type = "string",
			name = "entity_name",
			single = true
		},
		{
			type = "vector",
			name = "pos",
			single = true
		},
	},
	callback = function(ctx, args, options)
		local success
		if options.count then
			for _ = 1, options.count do
				success = core.add_entity(args.pos, args.entity_name)
			end
		else
			success = core.add_entity(args.pos, args.entity_name)
		end

		if not success then
			error("couldnt summon entity")
		end
	end,
})

lush.Command:new(
{
	name = "pulverize",
	description = "clears items",
	privs = {clear = true},
	arguments =
	{
		{
			name = "items",
			type = "inventory_item_list",
			single = true
		},
	},
	callback = function(ctx, args, options)
		local lists = args.items:get_lists()

		for _, stacks in pairs(lists) do
			for _, stack in pairs(stacks) do
				stack:clear()
			end
		end

		args.items:set_lists(lists)
	end,
})

lush.Command:new(
{
	name = "fix_items",
	description = "fixes items",
	privs = {fix = true},
	arguments =
	{
		{
			name = "items",
			type = "inventory_item_list",
			single = true
		},
	},
	callback = function(ctx, args, options)
		local lists = args.items:get_lists()

		for _, stacks in pairs(lists) do
			for _, stack in pairs(stacks) do
				stack:set_wear(0)
			end
		end

		args.items:set_lists(lists)
	end,
})

lush.Command:new(
{
	name = "itemmeta",
	description = "prints item metadata",
	privs = {debug = true},
	arguments =
	{
		{
			name = "items",
			type = "inventory_item_list",
			single = true
		},
	},
	callback = function(ctx, args, options)
		local lists = args.items:get_lists()

		for _, stacks in pairs(lists) do
			for _, stack in pairs(stacks) do
				ctx.stdout:push(stack:get_meta():to_table())
			end
		end
	end,
})

lush.Command:new(
{
	name = "setitemmeta",
	description = "sets item metadata",
	privs = {give = true},
	arguments =
	{
		{
			name = "items",
			type = "inventory_item_list",
			single = true
		},
		{
			name = "meta",
			type = "table",
			single = true
		},
	},
	callback = function(ctx, args, options)
		local lists = args.items:get_lists()

		for _, stacks in pairs(lists) do
			for _, stack in pairs(stacks) do
				stack:get_meta():from_table({fields = args.meta})
			end
		end

		args.items:set_lists(lists)
	end,
})

lush.Command:new(
{
	name = "lsitems",
	description = "lists items",
	options = {["inventory-format"] = false},
	arguments =
	{
		{
			name = "items",
			type = "inventory_item_list",
			single = true
		},
	},
	callback = function(ctx, args, options)

		if options["inventory-format"] then
				for list_name, list in pairs(args.items:get_lists()) do
					ctx.stdout:push("List " .. list_name)
					for _, v in pairs(list) do
						ctx.stdout:push(v:to_string())
					end
					ctx.stdout:push("EndInventoryList")
				end
		else
			-- print in tree like format
			-- the code looks funny because the last element must start with
			-- an "└─ " even if the last item in the inventory is empty
			-- this is a very clever solution to this problem
			for list_name, list in pairs(args.items:get_lists()) do
				ctx.stdout:push(list_name)

				local previous_item
				for _, item in pairs(list) do

					if not item:is_empty() then
						if previous_item then
							ctx.stdout:push("├─ " .. previous_item:to_string())
						end
						previous_item = item
					end
				end

				if previous_item then
					ctx.stdout:push("└─ " .. previous_item:to_string())
				end
			end
		end
	end,
})

lush.Command:new(
{
	name = "dump_luaentity",
	description = "prints the lua entity of a given object",
	privs = {debug = true},
	arguments =
	{
		{
			name = "object",
			type = "objectref",
			single = true
		},
	},
	callback = function(ctx, args, options)
		if args.object:get_luaentity() then
			ctx.stdout:push(args.object:get_luaentity())
		else
			error("couldnt get lua entity")
		end
	end,
})

lush.Command:new(
{
	name = "dump",
	description = "uses dump() on each element in stdin",
	arguments = {},
	callback = function(ctx, args, options)
		for _, v in ctx.stdin:iterator() do
			ctx.stdout:concat(lush.helpers.string_split(dump(v), "\n", false))
		end
	end,
})

lush.Command:new(
{
	name = "log",
	description = "dumps STDIN into the debug file for inspection",
	privs = {debug = true},
	arguments = {},
	stdin_type = "string",
	callback = function(ctx, args, options)
		core.log("action", string.format("%s dumped information:\n%s",
			ctx.env:get("name"), table.concat(ctx.stdin.container, "\n")))
	end,
})

