-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

--- @classmod LanguageConstruct_DoubleQuoteString
--	Describes a string enclosed in double quotes, which allows for parameter expansion
local ParameterExpansion
local max_chars_in_table = 10000

local DoubleQuotedString =
{
	instance_of = "LanguageConstruct_DoubleQuoteString",
}

DoubleQuotedString.__index = DoubleQuotedString

function DoubleQuotedString:new(t)
	if t == nil then
		t = {}
	end

	-- {{{ type checking
	assert(type(t) == "table", "t should be of type table. instead of " .. type(t))
	if t.text and type(t.text) ~= "string" then error("t.text should be of type string. instead of " .. type(t.text)) end
	-- }}}

	self = setmetatable(
	{
		elements = t.elements or {}
	}, DoubleQuotedString)
	self.__index = self

	return self
end

function DoubleQuotedString:evaluate(ctx)
	-- {{{ type checking
	assert(ctx.instance_of == "ShellContext", "ctx should be an instance of ShellContext. instead of " .. ctx.instance_of)
	-- }}}

	if #self.elements == 0 then
		return self.elements[1]
	end

	local output_string = ""

	for _, v in pairs(self.elements) do
		if type(v) == "string" then
			output_string = output_string .. v
		elseif type(v) == "table" and v.instance_of == "LanguageConstruct_ParameterExpansion" then
			output_string = output_string .. v:evaluate(ctx)
		else
			error("unexpected type")
		end
	end

	return output_string
end

function DoubleQuotedString:static_eval()
	return self.elements[1]
end

function DoubleQuotedString:parse(parser_ctx)
	-- {{{ type checking
	assert(parser_ctx.instance_of == "ParserContext", "parser_ctx should be an instance of ParserContext. instead of " .. parser_ctx.instance_of)
	-- }}}

	parser_ctx:advance()

	-- state: "*

	-- in lua the string concatentation operator is extreamly slow
	-- so instead of using it here, i append them to the char_table list
	-- and then table.concat() the list to get the final string
	--
	-- max_chars_in_table is the maximum of characters allowed in the table
	-- so that it wont use so much memory
	local char = parser_ctx:consume()
	local char_table = {}
	local char_table_index = 1
	local is_escaped = false
	local current_element = ""
	local parameter_expansion_instance

	while not (char == [["]] and not is_escaped) and not parser_ctx:is_EOF() do
		if not is_escaped and char == "$" then
			is_escaped = false

			parser_ctx:rewind() -- rewinding because the $ character is already consumed
			ParameterExpansion = ParameterExpansion or require("language_constructs.parameter_expansion")

			current_element = current_element .. table.concat(char_table)
			char_table = {}
			char_table_index = 1
			table.insert(self.elements, current_element)
			current_element = ""

			parameter_expansion_instance = ParameterExpansion:new()
			parameter_expansion_instance:parse(parser_ctx)

			table.insert(self.elements, parameter_expansion_instance)
		elseif not is_escaped and char == "\\" then
			is_escaped = true
		else
			is_escaped = false

			char_table[char_table_index] = char
			char_table_index = char_table_index + 1

			if char_table_index > max_chars_in_table then
				current_element = current_element .. table.concat(char_table)
				char_table = {}
				char_table_index = 1
			end
		end

		char = parser_ctx:consume()
	end

	current_element = current_element .. table.concat(char_table)

	-- handling edgecase where string ends with an expansion
	table.insert(self.elements, current_element)

	if #self.elements == 1 then
		self.static = true
	end

	-- state: "<string>"*
end

function DoubleQuotedString:dump(dump_ctx)
	local prefix = dump_ctx:color("(DoubleQuotedString) ", "ConstructSpecifier")
	if #self.elements == 1 then
		dump_ctx:write_text(string.format([[%s"%s"]], prefix, dump_ctx:color(self.elements[1], "String")))
	else
		dump_ctx:write_text(prefix)
		dump_ctx:write_text("{")
		dump_ctx:indent(1)
		for _, v in pairs(self.elements) do
			if type(v) == "string" then
				dump_ctx:write_line(dump_ctx:color(v, "String") .. ",")
			else
				v:dump(dump_ctx)
			end
		end
		dump_ctx:indent(-1)
		dump_ctx:write_text("}")
	end
end

return DoubleQuotedString
