-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

local helpers = require("helpers")
local CommandInvocation
local ShellContext
local InlineLua
local ParameterAsignment
local TypedList
local ForLoop
local WhileLoop
local IfStatement
local FunctionDefinition
local Expression


local control_operators =
{
	sequential = 1,
	pipe = 2,
	success = 3,
	failure = 4,
	iterator_pipe = 5,
}

local Shell =
{
	instance_of = "LanguageConstruct_Shell"
}

Shell.__index = Shell

function Shell:new(t)
	if t == nil then
		t = {}
	end

	-- {{{ type checking
	assert(type(t) == "table", "t should be of type table. instead of " .. type(t))
	-- }}}

	self = setmetatable(
	{
		--- list where the first elements is a language construct (like a command invocation)
		--	and the elements afterwards is an enum type that signifies the operator (pipe, or, and, ; )
		--	this is kinda a weird, but its the most straightforward way
		actions = t.actions or {}
	}, Shell)

	return self
end

-- options have these fields
-- share_environment - bool - if true, then the environment is shared between the two
-- subshell_ctx - SubshellContext - the context used by the shell. Useful if you're calling a shell multiple times
-- and want to recycle the subshell context, incompatible with share_environment
-- complete_callback - function - callback run when the shell finishes
function Shell:evaluate(ctx, options)
	-- {{{ type checking
	assert(ctx.instance_of == "ShellContext", "ctx should be an instance of ShellContext. instead of " .. ctx.instance_of)
	-- }}}

	local i = 1
	local subshell_ctx
	local complete_callback = options and options.complete_callback or nil

	local iterator_pipe_list
	local iterator_pipe_type
	local iterator_pipe_stdout

	-- forward definition. Necessary for the run_subshell function to be accesible in run_iterator_pipe, but be defined after
	local run_subshell

	-- assumes that iterator_pipe_list, iterator_pipe_type and iterator_pipe_stdout are set to appopriate values.
	-- I know... its shit, but deffered eexecution sucks like that
	local function run_iterator_pipe(iterator_pipe_idx)
		local success, error_message
		subshell_ctx.stdin = TypedList:new({type = iterator_pipe_type})
		-- subshell_ctx.stdout:clear()
		while iterator_pipe_idx <= #iterator_pipe_list do
			subshell_ctx.env:set("stdin", iterator_pipe_list[iterator_pipe_idx], false, true)

			if self.actions[i].instance_of == "LanguageConstruct_Shell" then
				-- special behaviour so subshells share the environment
				subshell_ctx:interrupt()
				success, error_message = pcall(self.actions[i].evaluate, self.actions[i], subshell_ctx,
				{
					share_environment = true,
					complete_callback = function()
						subshell_ctx:stop_interruption()
					end
				})

			else
				success, error_message = pcall(self.actions[i].evaluate, self.actions[i], subshell_ctx)
			end

			if not success then
				core.debug("ERR2", dump(error_message))
			end

			iterator_pipe_stdout:concat(subshell_ctx.stdout)
			subshell_ctx.stdout:clear()

			iterator_pipe_idx = iterator_pipe_idx + 1

			if subshell_ctx.interrupted then
				return subshell_ctx:add_interruption_callback(function()
					run_iterator_pipe(iterator_pipe_idx)
				end)
			end
		end

		ctx.stdout:concat(iterator_pipe_stdout.container)
		subshell_ctx.stdout:clear()
		subshell_ctx.stdin:clear()
		subshell_ctx.env:unset("stdin")

		i = i + 2
		run_subshell()
	end

	run_subshell = function()
		local success, error_message
		local operator
		while i <= #self.actions do
			operator = self.actions[i - 1] or control_operators.sequential
			if operator == control_operators.iterator_pipe then
				iterator_pipe_list = subshell_ctx.stdout.container
				iterator_pipe_type = subshell_ctx.stdout.type
				iterator_pipe_stdout = TypedList:new()
				subshell_ctx.stdout = TypedList:new()

				return run_iterator_pipe(1)
			elseif operator == control_operators.sequential
			or (subshell_ctx.exit_status ~= 0 and operator == control_operators.failure)
			or (subshell_ctx.exit_status == 0 and operator ~= control_operators.failure) then

				if operator == control_operators.pipe then
					subshell_ctx.stdin = subshell_ctx.stdout
					subshell_ctx.stdout = TypedList:new()
				else
					ctx.stdout:concat(subshell_ctx.stdout)
					subshell_ctx.stdin:clear()
					subshell_ctx.stdout:clear()
				end

				ctx.env.parameters["?"] = subshell_ctx.exit_status
				ctx.env.parameter_attributes["?"] = 3

				if self.actions[i].instance_of == "LanguageConstruct_Shell" then
					-- special behaviour so subshells share the environment
					subshell_ctx:interrupt()
					success, error_message = pcall(self.actions[i].evaluate, self.actions[i], subshell_ctx,
					{
						share_environment = true,
						complete_callback = function()
							subshell_ctx:stop_interruption()
						end
					})
				else
					success, error_message = pcall(self.actions[i].evaluate, self.actions[i], subshell_ctx)
				end

				if success then
					subshell_ctx.exit_status = 0
					subshell_ctx.stderr = ""
				else
					subshell_ctx.exit_status = 1
					subshell_ctx.stderr = error_message
					ctx.stdout:push(error_message)
				end

			end
			i = i + 2

			if subshell_ctx.interrupted then
				-- MY (above averge) IQ IS BEYOND YOUR COMPREHENSION
				return subshell_ctx:add_interruption_callback(
				function()
					run_subshell()
				end)
			end
		end

		ctx.stdout:concat(subshell_ctx.stdout)
		ctx.stderr = subshell_ctx.stderr
		ctx.exit_status = subshell_ctx.exit_status

		if complete_callback then
			complete_callback(true, nil, ctx)
		end
	end

	TypedList = TypedList or require("typed_list")
	ShellContext = ShellContext or require("shell_context")

	if options and options.subshell_context then
		subshell_ctx = options.subshell_context
	else
		subshell_ctx = ShellContext:new()
		subshell_ctx.stdin:concat(ctx.stdin)
		subshell_ctx.privilege_cache = ctx.privilege_cache
	end

	if options and options.share_environment then
		subshell_ctx.env = ctx.env
	else
		subshell_ctx.env:inherit(ctx.env)
	end

	run_subshell()
end

local terminator_cache = {}
local default_terminators = helpers.create_trie({";", "|", "||", "&&", ">/", "\n", "//"})
local newline_set = {["\n"] = true}
local lvalue_trie = helpers.create_trie({"="})

local subshell_terminators = helpers.create_trie({")"})
local combined_terminators_cache = {}

-- terminators is a trie of terminators for each action
-- shell_terminators is a trie of terminators for the while shell
function Shell:parse(parser_ctx, terminators, shell_terminators)
	-- {{{ type checking
	assert(parser_ctx.instance_of == "ParserContext", "parser_ctx should be an instance of ParserContext. instead of " .. tostring(parser_ctx.instance_of))
	-- }}}

	CommandInvocation = CommandInvocation or require("language_constructs.command_invocation")
	InlineLua = InlineLua or require("language_constructs.inline_lua")
	ParameterAsignment = ParameterAsignment or require("language_constructs.parameter_asignment")
	ForLoop = ForLoop or require("language_constructs.for_loop")
	IfStatement = IfStatement or require("language_constructs.if_statement")
	WhileLoop = WhileLoop or require("language_constructs.while_loop")
	FunctionDefinition = FunctionDefinition or require("language_constructs.function_definition")

	if terminator_cache[terminators] then
		terminators = terminator_cache[terminators]
	else
		local tmp = helpers.add_tries(default_terminators, terminators)
		terminator_cache[terminators] = tmp
		terminators = tmp
	end

	local combined_terminators -- the terminators for both the shell, and the command

	if combined_terminators_cache[terminators] then
		if combined_terminators_cache[terminators][shell_terminators] then
			combined_terminators = combined_terminators_cache[terminators][shell_terminators]
		else
			local tmp = helpers.add_tries(terminators, shell_terminators)
			combined_terminators = tmp
			combined_terminators_cache[terminators][shell_terminators] = tmp
		end
	else
		local tmp = helpers.add_tries(terminators, shell_terminators)
		combined_terminators_cache[terminators] = {}
		combined_terminators_cache[terminators][shell_terminators] = tmp
		combined_terminators = tmp
	end

	local action
	local operator
	local char
	while not parser_ctx:is_EOF() do

		local offset = 1
		parser_ctx:consume_whitespaces(true)
		while parser_ctx:peek(offset) == "/" and parser_ctx:peek(offset + 1) == "/" do
			-- If the line/action starts with a comment, seek to the start of the next line.
			-- That newline will then be consumed by the line just below the if statment
			parser_ctx:skip_until(newline_set)
			parser_ctx:advance()
			parser_ctx:consume_whitespaces(true)
		end

		parser_ctx:consume_whitespaces(true)
		local matched, _ = parser_ctx:match(shell_terminators)

		if matched then
			break
		end

		char = parser_ctx:peek()

		local instance_table = {}
		self._temp = instance_table

		if char == "(" then -- expected: subshell
			parser_ctx:advance()
			action = Shell:new(instance_table)
			action:parse(parser_ctx, terminators, subshell_terminators)
		elseif char == "`" then -- expected: inline lua code
			action = InlineLua:new(instance_table)
			action:parse(parser_ctx)
		elseif string.find(parser_ctx.text, "^for", parser_ctx.character_index) then -- expected: for loop
			action = ForLoop:new(instance_table)
			action:parse(parser_ctx)
		elseif string.find(parser_ctx.text, "^if", parser_ctx.character_index) then -- expected: if statement
			action = IfStatement:new(instance_table)
			action:parse(parser_ctx)
		elseif string.find(parser_ctx.text, "^while", parser_ctx.character_index) then -- expected: while loop
			action = WhileLoop:new(instance_table)
			action:parse(parser_ctx)
		elseif string.find(parser_ctx.text, "^function", parser_ctx.character_index) then
			action = FunctionDefinition:new(instance_table)
			action:parse(parser_ctx)
		else
			-- partial parsing. Parsing the expression first, and then checking the character it terminated with. If the
			-- character is an `=`, then the expression is a paramter asignment, otherwise its a command

			-- To avoid the parameter asignment and command language constructs from parsing the first expression again, set
			-- their first expressions to the value we just parsed, and add logic so that if those members are set, the language
			-- constructs will start parsing from that point
			Expression = Expression or require("language_constructs.expression")
			local expression = Expression:new()
			local yet_another_terminator_trie
			if terminator_cache[combined_terminators] then
				yet_another_terminator_trie = terminator_cache[combined_terminators]
			else
				local tmp = helpers.add_tries(combined_terminators, lvalue_trie)
				yet_another_terminator_trie = tmp
				terminator_cache[combined_terminators] = tmp
			end

			expression:parse(parser_ctx, yet_another_terminator_trie)

			if parser_ctx:peek() == "=" then
				instance_table.lvalue = expression
				action = ParameterAsignment:new(instance_table)
				action:parse(parser_ctx, terminators)
			else
				instance_table.command = expression
				action = CommandInvocation:new(instance_table)
				action:parse(parser_ctx, combined_terminators)
			end
		end

		table.insert(self.actions, action)

		-- state: <previous action if any><action terminator><whitespace><action>*

		parser_ctx:consume_whitespaces()
		matched, operator = parser_ctx:match(terminators, true)
		parser_ctx:consume_whitespaces()

		-- if not a control operator
		if not matched and not parser_ctx:is_EOF() then
			-- checking if the shell should be terminated by a keyword
			parser_ctx:consume_whitespaces(true)
			local matched, _ = parser_ctx:match(shell_terminators, true)

			if matched then
				table.insert(self.actions, control_operators.sequential)
				break
			end

			-- if the parser got to this point
			-- this means that a the action that was just parsed ended parsing in an unexpected location
			-- this means everything broke
			error("OH SHIT OH FUCK!!!!!\n" .. action.instance_of .. " fucked up BIG time")
		end

		-- state: <previous iteration><whitespaces><action><control operator>*

		if operator == "//" then
			parser_ctx:skip_until(newline_set)
		end

		if operator == "|" then
			table.insert(self.actions, control_operators.pipe)
		elseif operator == "&&" then
			table.insert(self.actions, control_operators.success)
		elseif operator == ">/" then
			table.insert(self.actions, control_operators.iterator_pipe)
		elseif operator == "||" then
			table.insert(self.actions, control_operators.failure)
		else
			table.insert(self.actions, control_operators.sequential)
		end

		-- checking if the shell should be terminated by a keyword
		parser_ctx:consume_whitespaces(true)
		local matched, _ = parser_ctx:match(shell_terminators)

		-- state: <previous iteration><whitespaces><action><control operator><whitespaces><terminator>*
		if matched then
			break
		end
	end

	assert(#self.actions ~= 0, "empty shell is not allowed")
	--	handling an edgecase where a shell ends with a control operator, like
	--	echo text |
	--	but allow something like this
	--	echo text;
	assert(self.actions[#self.actions] == control_operators.sequential, "unexpected control operator")
end

local operator_names =
{
	"<Sequential>",
	"<Pipe>",
	"<Success>",
	"<Failure>",
	"<Iterator pipe>",
}

function Shell:dump(dump_ctx)
	dump_ctx:write_text(dump_ctx:color("(Shell)", "ConstructSpecifier"))
	dump_ctx:new_line()
	dump_ctx:write_text("[")
	dump_ctx:indent(1)
	dump_ctx:new_line()

	local i = 1

	while i <= #self.actions do
		local operator = self.actions[i - 1] or control_operators.sequential
		dump_ctx:write_text(operator_names[operator])
		dump_ctx:new_line(1)
		self.actions[i]:dump(dump_ctx)
		i = i + 2
		dump_ctx:new_line(1)
	end

	dump_ctx:indent(-1)
	dump_ctx:new_line()
	dump_ctx:write_text("]")
end

return Shell
