-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

-- @classmod ParserContext <br>
-- to be honest, i have no expierience in making handmade parsers and i thought this was a good learning expierience <br>
-- (thats why i didnt use a generator) <br>
-- so dont be mad if some things arent understandable <br>
-- here is the overall plan how it works: <br>
-- the ParserContext class is a wrapper around the input text that provides peek() and consume() methods <br>
-- when parsing the parser attempts to recognize what language construct is next and calls parse() method of the appopriate LanguageConstruct derivative classes <br>
-- that does all the parsing and returns control to the parser in a different location of the parsed text <br>

-- @type ParserContext
local ParserContext =
{
	instance_of = "ParserContext",
}

ParserContext.__index = ParserContext

-- constructor
-- @tab t table with matching keys as the class, with initialization values for the members
-- @treturn ParserContext the newly created Command instance
function ParserContext:new(t)
	if t == nil then
		t = {}
	end

	-- {{{ type checking
	assert(type(t) == "table", "t should be of type table. instead of " .. type(t))
	assert(not t.character_index or type(t.character_index) == "number", "t.character_index should be of type number. instead of " .. type(t.character_index))
	assert(not t.text or type(t.text) == "string", "t.text should be of type string. instead of " .. type(t.text))
	-- }}}

	self = setmetatable(
	{
		-- @number[opt=1] value specifying the index of next character to be read
		character_index = t.character_index or 1,
		-- @string[opt=""] text to be parser
		text = t.text or ""

	}, ParserContext)

	return self
end

-- returns the next nth character from the current position
-- @number[opt=1] offset the amount of characters to advance
-- @treturn string a single character
-- @see ParserContext:consume
function ParserContext:peek(offset)
	offset = offset or 1

	local offset_index = self.character_index + offset - 1
	return string.sub(self.text, offset_index, offset_index)
end

-- returns the next nth character from the current position and updates the current character index to that position
-- @number[opt=1] offset the amount of characters to advance
-- @treturn string a single character
-- @see ParserContext:peek, ParserContext:rewind
function ParserContext:consume(offset)
	offset = offset or 1

	self.character_index = self.character_index + offset

	return string.sub(self.text, self.character_index - 1, self.character_index - 1)
end

-- returns to the previous nth position
-- @number[opt=1] offset the amount of characters to rewind
-- @treturn nil
-- @see ParserContext:consume
function ParserContext:rewind(offset)
	offset = offset or 1

	self.character_index = self.character_index - offset
end

function ParserContext:advance(offset)
	offset = offset or 1

	self.character_index = self.character_index + offset
end

-- consumes characters until it finds one that is in the terminator set (or the end of string),
--	if the current character is in the terminator set, does nothing
--	@tab terminator_set a set of characters
--	@bool inverse if true then it will skip to the first character that isnt in the terminator set
--	@treturn nil
function ParserContext:skip_until(terminator_set, inverse)
	-- {{{ type checking
	assert(type(terminator_set) == "table", "terminator_set should be of type table. instead of " .. type(terminator_set))
	assert(not inverse or type(inverse) == "boolean", "inverse should be of type boolean. instead of " .. type(inverse))
	-- }}}

	-- this is a micro optimization, but since the function is used so often
	-- this saves a considerable amount of time
	if terminator_set[self:peek()] ~= inverse then
		return
	end

	local offset = 1
	while terminator_set[self:peek(offset)] == inverse and self.character_index + offset <= #self.text + 1 do
		offset = offset + 1
	end

	if offset ~= 1 then
		self.character_index = self.character_index + offset - 1
	end
end

local white_char_set = {[" "] = true, ["\t"] = true}
local white_char_set_plis_newlines = {[" "] = true, ["\t"] = true, ["\n"] = true}

function ParserContext:consume_whitespaces(consume_newlines)
	local set = consume_newlines and white_char_set_plis_newlines or white_char_set
	while set[string.sub(self.text, self.character_index, self.character_index)] do
		self.character_index = self.character_index + 1
	end
end

-- tries to match any of the specified terminators, starting from the current position and returns the longest match <br>
--	@tab a trie of terminators. See helpers.create_trie
--	@bool consume whether to consume the matched terminator
--	@return the frst return value a bool which says whether any terminator matched, if so then the second value is the terminator that matched, if no terminator matched, then the second value is nil
function ParserContext:match(terminators, consume)
	local i = 0
	local char = string.sub(self.text, self.character_index, self.character_index)

	local head = terminators[char]
	local longest_terminator

	-- first check if the first character even matched. Its a very common case it dosen't, so it makes sense to optimize that
	if head then
		repeat
			if head[1] ~= nil then
				longest_terminator = head[1]
				if head[2] then
					i = i + 1
					break
				end
			end

			char = string.sub(self.text, self.character_index + i + 1, self.character_index + i + 1)
			i = i + 1

			head = head[char]
		until not head
	end

	if longest_terminator then
		if consume then
			self.character_index = self.character_index + i
		end

		return true, longest_terminator
	else
		return false, nil
	end
end

-- similar to ParserContext:match but only supports a single terminal.
-- This is has less overhead
-- @string terminal
-- @treturn bool whether sucessfully matched
-- TODO: investigate the fact that consume does nothing
function ParserContext:match_fast(terminal, consume)
	local index = 1

	-- minetest.debug("BAM")
	while index < #terminal do
		if self:peek(index) ~= terminal:sub(index, index) then
			-- print(string.format("invalidc char: '%s' in index: %s", self:peek(index), index))
			-- minetest.debug(index, self:peek(index), terminal:sub(index, index))
			-- minetest.debug(self.character_index)
			return false
		end

		index = index + 1
	end

	self.character_index = self.character_index + index

	return true
end

-- returns true if is at the end of the string
function ParserContext:is_EOF()
	return self.character_index > #self.text
end

-- consumes a character and errors if the character isn't the expected character
-- @sring expected_char the expected character
function ParserContext:expect(expected_char)
	local char = self:consume()
	assert(expected_char == char, string.format([[unexpected character: expected "%s" but got "%s"]], expected_char, char))
end

-- method called when an error occurs while parsing
-- @string message the error message
-- treturn nil
function ParserContext:error(message)
	print(tostring(self.character_index) .. ": " .. tostring(message))
end

return ParserContext
