-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

require("tests.busted_definitions")
require("tests.command_definitions")

describe("LanguageConstructs_IfStatement class: ",
function()
	local IfStatement = require("language_constructs.if_statement")
	local ParserContext = require("parser_context")
	local ShellContext = require("shell_context")
	local Command, Registered_Commands = require("command")
	local helpers = require("helpers")
	local parser_context_instance
	local shell_context_instance
	local if_statement_instance

	before_each(
	function()
		parser_context_instance = ParserContext:new()
		shell_context_instance = ShellContext:new()
		shell_context_instance.env:inherit(lush.global_env)
		if_statement_instance = IfStatement:new()
	end)

	it("parses correctly when given a successful if statement",
	function()
		parser_context_instance.text = [[if 1 then echo "success"; end]]

		if_statement_instance:parse(parser_context_instance)
		if_statement_instance:evaluate(shell_context_instance)

		assert.equals(30, parser_context_instance.character_index)
		assert.equals("success", shell_context_instance.stdout:get(1))
	end)

	it("parses correctly when given a failed if statement",
	function()
		parser_context_instance.text = [[if 0 then echo "success"; end]]

		if_statement_instance:parse(parser_context_instance)
		if_statement_instance:evaluate(shell_context_instance)

		assert.equals(30, parser_context_instance.character_index)
		assert.equals(nil, shell_context_instance.stdout:get(1))
	end)

	it("parses correctly when given a failed if statment with an else",
	function()
		parser_context_instance.text = [[if 0 then echo "success"; else echo "fail"; end]]

		if_statement_instance:parse(parser_context_instance)
		if_statement_instance:evaluate(shell_context_instance)

		assert.equals(48, parser_context_instance.character_index)
		assert.equals("fail", shell_context_instance.stdout:get(1))
	end)

	it("parses correctly when given a successful if statment with an else",
	function()
		parser_context_instance.text = [[if 1 then echo "success"; else echo "fail"; end]]

		if_statement_instance:parse(parser_context_instance)
		if_statement_instance:evaluate(shell_context_instance)

		assert.equals(48, parser_context_instance.character_index)
		assert.equals("success", shell_context_instance.stdout:get(1))
	end)

	it("parses correctly when given a successful if statment with an elseif that succedes",
	function()
		parser_context_instance.text = [[if 0 then echo "success"; elseif 1 then echo "second success"; end]]

		if_statement_instance:parse(parser_context_instance)
		if_statement_instance:evaluate(shell_context_instance)

		assert.equals(67, parser_context_instance.character_index)
		assert.equals("second success", shell_context_instance.stdout:get(1))
	end)

	it("parses correctly when given a successful if statment with an elseif that fails",
	function()
		parser_context_instance.text = [[if 0 then echo "success"; elseif 0 then echo "second success"; end]]

		if_statement_instance:parse(parser_context_instance)
		if_statement_instance:evaluate(shell_context_instance)

		assert.equals(67, parser_context_instance.character_index)
		assert.equals(nil, shell_context_instance.stdout:get(1))
	end)

	it("parses correctly when given a successful if statment with an elseif that fails, plus an else",
	function()
		parser_context_instance.text = [[if 0 then echo "success"; elseif 0 then echo "second success"; else echo "fail" end]]

		if_statement_instance:parse(parser_context_instance)
		if_statement_instance:evaluate(shell_context_instance)

		assert.equals(84, parser_context_instance.character_index)
		assert.equals("fail", shell_context_instance.stdout:get(1))
	end)
end)
