-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

local lush_override_commands = core.settings:get_bool("lush_override_commands")

-- setting default value
if lush_override_commands == nil then
	lush_override_commands = true
end

if lush_override_commands then
	core.unregister_chatcommand("grantme")
	core.unregister_chatcommand("revokeme")
end

--- Update privs

local storage = core.get_mod_storage()

-- list of tables with format:
-- name - player name
-- priv - the privilege name
-- expiration_time - unix time after which the privilege is expired
-- type - "revoke" or "grant", whether the privliege was given or revoked temporarily
local temporary_privs = core.deserialize(storage:get("temporary_privs") or "") or {}

-- key-value:
-- key: player name
-- value: unix time of the expitation data
local bans = core.deserialize(storage:get("bans") or "") or {}

local function update_privs()
	local current_time = os.time()
	local player_privs
	local to_be_removed = {} -- table of indexes to be removed
	local temporary_privs_changed = false
	for i, data in pairs(temporary_privs) do
		if data.expiration_time < current_time and core.get_player_by_name(data.name) then
			if not core.registered_privileges[data.priv] then
				core.log("error", "trying to grant/revoke a nonexistent privilege while updating temporary privileges."
				.. "this could be caused by the mod not being loaded anymore, privilege in question: " .. data.priv)
			else
				temporary_privs_changed = true

				player_privs = core.get_player_privs(data.name)

				if data.type == "revoke" then
					player_privs[data.priv] = true
				else
					player_privs[data.priv] = nil
				end

				core.set_player_privs(data.name, player_privs)

				table.insert(to_be_removed, i)
			end
		end
	end

	if temporary_privs_changed then
		for _, v in pairs(to_be_removed) do
			table.remove(temporary_privs, v)
		end

		storage:set_string("temporary_privs", core.serialize(temporary_privs))
	end

	local bans_changed
	for name, expiration_time in pairs(bans) do
		if expiration_time < current_time then
			bans_changed = true
			core.unban_player_or_ip(name)
			expiration_time = nil
		end
	end

	if bans_changed then
		storage:set_string("bans", core.serialize(bans))
	end

	core.after(tonumber(core.settings:get("lush_priv_refresh_interval")) or 5.0, function()
		update_privs()
	end)
end

update_privs()

core.register_privilege("mute",
{
	description = "players with this privilege can mute others"
})

core.register_privilege("freeze",
{
	description = "Can use the freeze command"
})

lush.Command:new(
{
	name = "ban",
	description = "bans players",
	options =
	{
		["script"] = false,
	},
	privs = {ban = true},
	arguments =
	{
		{
			name = "players",
			type = "player_name",
			single = false
		}
	},
	callback = function(ctx, args, options)
		if core.is_singleplayer() then
			error("cant ban players in singleplayer")
		end

		local success
		local successful_bans = {}
		local failed_bans = {}
		for _, name in pairs(args.players) do
			success = core.ban_player(name)

			if success then
				bans[name] = os.time() + args.time
				table.insert(successful_bans, name)
			else
				table.insert(failed_bans, name)
			end
		end

		if #successful_bans == 0 then
			error("failed to ban players")
		end

		if options["script"] then
			ctx.stdout:concat(successful_bans)
		else
			if #failed_bans ~= 0 then
				ctx.stdout:push(string.format("failed to ban players: %s", table.concat(failed_bans, ", ")))
			end
		end

		core.log("action", string.format("%s banned players: %s", ctx.env:get("name"), table.concat(successful_bans, ", ")))
		storage:set_string("bans", core.serialize(bans))
	end,
})

lush.Command:new(
{
	name = "unban",
	description = "unbans players",
	options = {
		["script"] = false,
	},
	privs = {ban = true},
	arguments =
	{
		{
			name = "players",
			type = "player_name",
			single = false
		},
	},
	callback = function(ctx, args, options)
		local success
		local successful_unbans = {}
		local failed_unbans = {}
		for _, name in pairs(args.players) do
			success = core.unban_player_or_ip(name)

			if success then
				table.insert(successful_unbans, name)
			else
				table.insert(failed_unbans, name)
			end
		end

		if #successful_unbans == 0 then
			error("failed to unban players")
		end

		if options["script"] then
			ctx.stdout:concat(successful_unbans)
		else
			if #failed_unbans ~= 0 then
				ctx.stdout:push(string.format("failed to unban players: %s", table.concat(failed_unbans, ", ")))
			end
		end

		core.log("action", string.format("%s unbanned players: %s", ctx.env:get("name"), table.concat(successful_unbans, ", ")))
		storage:set_string("bans", core.serialize(bans))
	end,
})

lush.Command:new(
{
	name = "lsbans",
	description = "lists all banned players",
	privs = {ban = true},
	arguments = {},
	callback = function(ctx, args, options)
		ctx.stdout:concat(core.get_ban_list())
	end,
})

lush.Command:new(
{
	name = "kick",
	description = "kicks player out of the server",
	options = {
		["script"] = false
	},
	privs = {ban = true},
	arguments =
	{
		{
			name = "players",
			type = "player_name",
			single = false,
		},
		{
			name = "reason",
			type = "string",
			single = true
		},
	},
	callback = function(ctx, args, options)
		if core.is_singleplayer() then
			error("cant kick players in singleplayer")
		end

		local success
		local successful_kicks = {}
		local failed_kicks = {}
		for _, name in pairs(args.players) do
			success = core.kick_player(name, args.reason)

			if success then
				table.insert(successful_kicks, name)
			else
				table.insert(failed_kicks, name)
			end
		end

		if #successful_kicks == 0 then
			error("failed to kick players")
		end

		if options["script"] then
			ctx.stdout:concat(successful_kicks)
		else
			if #failed_kicks ~= 0 then
				ctx.stdout:push(string.format("failed to ban players: %s", table.concat(failed_kicks, ", ")))
			end
		end

		core.log("action", string.format("%s banned players: %s", ctx.env:get("name"), table.concat(successful_kicks, ", ")))
	end,
})

lush.Command:new(
{
	name = "freeze",
	description = "freezes a player, which prevents them from moving or interacting",
	privs = {freeze = true},
	arguments =
	{
		{
			name = "players",
			type = "player_name",
			single = false,
		},
		{
			name = "duration",
			type = "time",
			optional = true,
			single = true,
		},
	},
	callback = function(ctx, args, options)
		local player_privs
		for _, name in args.players:iterator() do
			player_privs = core.get_player_privs(name)
			player_privs["interact"] = nil
			core.set_player_privs(name, player_privs)
			table.insert(temporary_privs,
			{
				name = name,
				expiration_time = os.time() + (args.duration or math.huge),
				priv = "interact",
				type = "revoke"
			})

			local player = core.get_player_by_name(name)

			player:add_velocity(-player:get_velocity())
			player:set_physics_override(
				{speed = 0, speed_walk = 0, speed_crouch = 0, speed_climb = 0, jump = 0, gravity = 0.001, acceleration_air = 0.0001})

			if args.duration then
				core.after(args.duration, function()
					core.get_player_by_name(name):set_physics_override(
						{speed = 1, speed_walk = 1, speed_crouch = 1, speed_climb = 1, jump = 1, gravity = 1, acceleration_air = 1})
				end)
			end
		end
	end,
})

lush.Command:new(
{
	name = "grant",
	description = "grants privileges to player",
	options = {["time"] = true},
	arguments =
	{
		{
			name = "player",
			type = "player_name",
			single = true
		},
		{
			rule_type = "terminal",
			value =
			{
				["privs"] = true,
				["privileges"] = true,
			},
			consume = true,
			optional = true,
		},
		{
			rule_type = "block",
			name = "privs",
			type = "string",
			required = true
		},
	},
	callback = function(ctx, args, options)
		local caller_privs = core.get_player_privs(ctx.env:get("name"))

		if not (caller_privs.privs or caller_privs.basic_privs) then
			error("insufficient privileges")
		end

		local player_privs = core.get_player_privs(args.player)
		local unknown_privs = {}
		local basic_privs = core.string_to_privs(core.settings:get("basic_privs") or "interact,shout")
		local length

		if options["time"] then
			if type(options["time"]) ~= "string" and type(options["time"]) ~= "number" then
				error("--time option requires the time as an argument")
			end

			if type(options["time"]) == "number" then
				length = options["time"]
			else
				length = lush.helpers.parse_time(options["time"])

				if not length then
					error("invalid time format for the --time option")
				end
			end
		end

		for _, priv in args.privs:iterator() do
			if not basic_privs[priv] and not caller_privs.privs then
				error("Your privileges are insufficient")
			end
			if not core.registered_privileges[priv] then
				if priv == "all" then
					for k, _ in pairs(core.registered_privileges) do
						player_privs[k] = true
					end
				else
					table.insert(unknown_privs, priv)
				end
			end

			player_privs[priv] = true

			if options["time"] then
				table.insert(temporary_privs,
				{
					name = args.player,
					priv = priv,
					expiration_time = os.time() + length,
					type = "grant"
				})
			end
		end

		if options["time"] then
			storage:set_string("temporary_privs", core.serialize(temporary_privs))
		end

		if #unknown_privs ~= 0 then
			error("unknown privileges: " .. table.concat(unknown_privs, ", "))
		end

		core.set_player_privs(args.player, player_privs)

		-- updating the privilege cache if changing privileges of himself
		if args.player == ctx.env:get("name") then
			ctx.privilege_cache = player_privs
		end

		for _, priv in pairs(args.privs) do
			-- call the on_grant callbacks
			core.run_priv_callbacks(args.player, priv, ctx.env:get("name"), "grant")
		end

		core.log("action", ctx.env:get("name") .. ' granted (' .. table.concat(args.privs, ", ") .. ') privileges to ' .. args.player)

		if args.player ~= ctx.env:get("name") then
			core.chat_send_player(args.player,
				ctx.env:get("name") .. " granted privileges" .. table.concat(args.privs, ", ") .. " to you")
		end
	end,
})

lush.Command:new(
{
	name = "revoke",
	description = "takes away privileges from a player",
	options = {["time"] = true},
	arguments =
	{
		{
			name = "player",
			type = "player_name",
			single = true
		},
		{
			rule_type = "terminal",
			value =
			{
				["privs"] = true,
				["privileges"] = true,
			},
			consume = true,
			optional = true,
		},
		{
			rule_type = "block",
			name = "privs",
			type = "string",
			required = true
		},
	},
	callback = function(ctx, args, options)
		local caller_privs = core.get_player_privs(ctx.env:get("name"))

		if not (caller_privs.privs or caller_privs.basic_privs) then
			error("insufficient privileges")
		end

		local player_privs = core.get_player_privs(args.player)
		local unknown_privs = {}
		local basic_privs = core.string_to_privs(core.settings:get("basic_privs") or "interact,shout")
		local is_admin = not core.is_singleplayer()
			and args.player == core.settings:get("name")
			and args.player ~= ""
		local priv_def
		local length

		if options["time"] then
			if type(options["time"]) ~= "string" and type(options["time"]) ~= "number" then
				error("--time option requires the time as an argument")
			end

			if type(options["time"]) == "number" then
				length = options["time"]
			else
				length = lush.helpers.parse_time(options["time"])

				if not length then
					error("invalid time format for the --time option")
				end
			end
		end

		for _, priv in args.privs:iterator() do
			if not basic_privs[priv] and not caller_privs.privs then
				error("Your privileges are insufficient")
			end

			priv_def = core.registered_privileges[priv]

			if not priv_def then
				if priv == "all" then
					for k, _ in pairs(player_privs) do
						player_privs[k] = nil
					end
				elseif not player_privs[priv] then
				-- its possible for players to have privleges 
				-- from mods that are no longer active
					table.insert(unknown_privs, priv)
				end
			elseif (core.is_singleplayer() and priv_def.give_to_singleplayer)
				or is_admin and priv_def.give_to_admin then
				error("you cant revoke " .. priv .. " from admin")
			end

			player_privs[priv] = nil

			if options["time"] then
				table.insert(temporary_privs,
				{
					name = args.player,
					priv = priv,
					expiration_time = os.time() + length,
					type = "revoke"
				})
			end
		end

		if options["time"] then
			storage:set_string("temporary_privs", core.serialize(temporary_privs))
		end

		if #unknown_privs ~= 0 then
			error("unknown privileges: " .. table.concat(unknown_privs, ", "))
		end

		core.set_player_privs(args.player, player_privs)

		-- updating the privilege cache if changing privileges of himself
		if args.player == ctx.env:get("name") then
			ctx.privilege_cache = player_privs
		end

		for _, priv in pairs(args.privs) do
			-- call the on_grant callbacks
			core.run_priv_callbacks(args.player, priv, ctx.env:get("name"), "revoke")
		end

		core.log("action", ctx.env:get("name") .. ' revoked (' .. table.concat(args.privs, ", ") .. ') privileges from ' .. args.player)

		if args.player ~= ctx.env:get("name") then
			core.chat_send_player(args.player,
				ctx.env:get("name") .. " granted privileges" .. table.concat(args.privs, ", ") .. " to you")
		end
	end,
})

lush.Command:new(
{
	name = "privs",
	description = "lists player's privileges",
	options = {},
	arguments =
	{
		{
			name = "player",
			type = "player_name",
			single = true
		}
	},
	callback = function(ctx, args, options)
		for priv, _ in pairs(core.get_player_privs(args.player)) do
			ctx.stdout:push(priv)
		end
	end,
})

lush.Command:new(
{
	name = "mute",
	description = "mutes players",
	privs = {mute = true},
	arguments =
	{
		{
			name = "player",
			type = "player_name",
			single = true
		},
		{
			name = "time",
			type = "time",
			single = true
		},
	},
	callback = function(ctx, args, options)
		local is_admin = not core.is_singleplayer()
			and args.player == core.settings:get("name")
			and args.player ~= ""

		if core.is_singleplayer() or is_admin then
			error("cant mute admins")
		end

		local privs = core.get_player_privs(args.player)
		privs["shout"] = nil
		core.set_player_privs(args.player, privs)

		table.insert(temporary_privs, {
			name = args.player,
			expiration_time = os.time() + args.time,
			priv = "shout",
			type = "revoke",
		})
	end,
})
