-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

-- @classmod Command
-- this class defines how commands are executed
local helpers = require("helpers")
local DatatypeValidator = require("datatype_validator")

local Registered_Commands = {}

local Command =
{
	instance_of = "Command",
}

Command.__index = Command

lush.defined_commands = 0
-- constructor <br>
--	note: dosent create a commands if there already exists a commands with the same name
--	@tab t definition table with matching keys as the class
--	@treturn Command|nil the newly created Command instance
--	@see helpers.is_option
function Command:new(t)
	lush.defined_commands = lush.defined_commands + 1
	t = t or {}

	-- {{{ type checking
		assert(type(t) == "table", "t should be of type table. instead of " .. type(t))
		if t.name and type(t.name) ~= "string" then error("name has to be of type string") end
		if t.options and type(t.options) ~= "table" then error("options has to be of type table") end
		if t.callback and type(t.callback) ~= "function" then error("name has to be of type function") end
	-- }}}

	self = setmetatable(
	{
		-- @string[opt=""] name name of the command
		name = t.name or "",

		-- @func[opt=nil]
		--	callback callback function that is called when the command is executed <br>
		--	should be called with call() method
		--  @ShellContext ctx shell context for the execution
		--  @tab args list of string arguments
		--  @tab options dictionary where the key is a string name of the option and the value is the argument
		--  (if applicable, otherwise is false)
		--  @treturn ShellContext shell context after execution
		--  @see Command:call
		callback = t.callback or nil,

		-- @tab[opt={}] options dictionary represending what options are allowed <br>
		-- key - string character <br>
		-- value - boolean value, whether it takes an argument <br>
		options = t.options or {},

		-- @tab[opt={}] arguments a list that describes what arguments the program expects <br>
		--	each element is a table with the following members: <br>
		--	name: the name of the argument, so it can be accessed by name instead of an index
		--	type: string or table of the datatypes it expects <br>
		--	single: whether is a single value or a TypedList
		--	terminator: string or a set of terminators that the argument should equal <br>
		--	consume_terminator: if true, then the terminator is removed from the argument list (defaults to true) <br>
		--
		--	if arguments isnt defined, then no argument checking is done
		 arguments = t.arguments or nil,

		 -- @string[opt=nil] the datatype the stdin is expected to be, or nil if no checking
		 stdin_type = t.stdin_type or nil
	}, Command)

	if not t.not_in_global_env then
		-- i know.. double not
		lush.global_env:set(self.name, self, true, true)
	end

	return self
end

-- splits arguments into options and executes the command
-- @ShellContext ctx shell context for the execution
-- @tab args list of arguments and options
-- @treturn ShellContext context after the execution
function Command:call(ctx, args)
	-- {{{ type checking
	assert(ctx.instance_of == "ShellContext", "ctx should be an instance of ShellContext. instead of " .. ctx.instance_of)
	-- }}}

	local options = {}
	local validated_args = {}
	local skip = false

	if self.stdin_type and ctx.stdin.type ~= self.stdin_type then
		local success = ctx.stdin:cast_to(self.stdin_type, ctx)

		assert(success, "stdin is incorrect type: expected \"" .. self.stdin_type .. "\" but got \"" .. DatatypeValidator.get_type_of(self.stdin) .. "\"")
	end

	-- maybe all this logic should be moved to CommandInvocation?
	--
	-- me months later: thats a stupid comment, wtf?!
	local is_valid_option, option_name
	for i, arg in ipairs(args) do
		if not skip then
			is_valid_option = false
			if type(arg) == "string" then
				is_valid_option, option_name = helpers.is_option(arg)
				if is_valid_option then
					assert(self.options[option_name] ~= nil, string.format("unrecognized option '%s'", option_name))

					if(self.options[option_name]) == true then
						-- if takes an argument

						-- if the option is the last argument, therefore there is no argument for the option passed
						assert(i ~= #args, string.format("option '%s' expects an argument", option_name))

						options[option_name] = args[i + 1]
						skip = true
					else
						options[option_name] = true
					end
				end
			end

			if not is_valid_option then
				table.insert(validated_args, arg)
			end
		else
			skip = false
		end
	end

	if self.arguments then
		validated_args = DatatypeValidator.validate(validated_args, self.arguments, ctx)
	end

	local success, message = pcall(self.callback, ctx, validated_args, options)

	if not success then
		core.debug("ERROR!", dump(message))
	end

	assert(success, message or "no error message provided")
	ctx.exit_status = 0

	return ctx
end

Command.__call = Command.call

-- unregisters the command and destroys the instance
--  @return nil
--  @see Command.is_name_registered
function Command:unregister()
	Registered_Commands[self.name] = nil
	self = nil
end

-- returns whether the command with the specified name is registered
--  @string name name of the command
--  @treturn bool whether the command is already registered
--  @see Command:unregister, Command:get_command_by_name
function Command.is_name_registered(name)
	return Registered_Commands[name] ~= nil
end

-- returns a registered command with the matching name
--  @string name name of the command
--  @treturn Command the command
--  @see Command:is_name_registered
function Command.get_command_by_name(name)
	return Registered_Commands[name]
end

-- creates an alias for a command
--	@string name
--	@string new_alias
--	@treturn nil
function Command.alias(name, new_alias)
	Registered_Commands[new_alias] = Registered_Commands[name]
end

return Command, Registered_Commands
