-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

-- @classmod Env <br>
-- container type which acts as a key/value storage for shell parameters
local helpers = require("helpers")

-- @type Env

local Env =
{
	instance_of = "Env",
}

Env.__index = Env

-- constructor
-- @treturn Env Env instance
function Env:new(t)
	t = t or {}

	self = setmetatable(
	{
		-- @tab parameters list containing all the defined parameters <br>
		--  key - string name of the parameter
		--  value - value of the parameter
		parameters = t.parameters or {},

		-- @tab read_only a set of keys for parameters that are read only
		-- read_only = t.read_only or {},

		-- @tab volatile_parameters same as Env.parameters, but isnt persistent over server restarts
		-- volatile_parameters = t.volatile_parameters or {},

		-- @Env an enviroment to inherit parameters from if they are undefined in local scope
		inherited_env = t.inherited_env,

		parameter_attributes = t.parameter_attributes or {}
	}, Env)

	return self
end

-- gets the value of a parameter with the given key
--	it will look for the parameter in the following order: <br>
--	first it checks whether the parameter is read only in any of the inherited environments, if so, returns the parameter value in the top most environment
--	otherwise it checks whether it exists as a volatile parameter, if not then it looks whether its a normal parameter, and if not then it tries to inherit it from other environments
--	@string key
--	@return value
function Env:get(key)
	-- {{{ type checking
	if key == nil then return nil end
	assert(type(key) == "string", "key should be of type string. instead of " .. type(key))
	-- }}}

	local output = self.parameters[key]

	if output == nil then
		output = self.inherited_env and self.inherited_env:get(key)
	end

	return output
end

-- sets a parameter with the given key and value,
-- optionally can also provide flags read_only and is_volatile
-- @string key
-- @param value
-- @bool read_only whether the value can be modified with Env:set
-- @bool is_volatile whether the parameter will be persistent over server reloads 
function Env:set(key, value, read_only, is_volatile)
	-- {{{ type checking
	assert(type(key) == "string", "key should be of type string. instead of " .. type(key))
	-- }}}

	local function set(env)
		env.parameter_attributes[key] = bit.bor(
			read_only and helpers.ParamAttributes.ReadOnly or 0,
			is_volatile and helpers.ParamAttributes.Volatile or 0
		)
		env.parameters[key] = value
	end
	assert(not self:is_param_readonly(key), "trying to overwrite a readonly value")

	local head = self

	repeat
		-- @todo add unit test for this case
		-- check parameter_attributes instead of parameters. In edge case when the parameter is nil, it should still set the
		-- inherited parameter
		if head.parameter_attributes[key] ~= nil then
			set(head)
			return
		end
		head = head.inherited_env
	until(head == nil)

	set(self)
end

-- removes a parameter with the given key
--	only works with persistent parameters
--	@string key
--	@treturn nil
function Env:unset(key)
	-- {{{ type checking
	assert(type(key) == "string", "key should be of type string. instead of " .. type(key))
	-- }}}

	if self.parameter_attributes[key] then
		self.parameters[key] = nil
		self.parameter_attributes[key] = nil
	elseif self.inherited_env then
		self.inherited_env:unset(key)
	end
end

-- checks whether the parameter with the given key is read only
--	@string key
--	@treturn bool
function Env:is_param_readonly(key)
	local head = self

	repeat
		if head.parameter_attributes[key]then
			return bit.band(head.parameter_attributes[key], helpers.ParamAttributes.ReadOnly) ~= 0
		end
		head = head.inherited_env
	until(head == nil)
	return (self.parameter_attributes[key] and bit.band(self.parameter_attributes[key], helpers.ParamAttributes.ReadOnly) ~= 0) or self.inherited_env and self.inherited_env:is_param_readonly(key)
	-- if self.read_only[key] then
	-- 	return true
	-- elseif self.inherited_env.instance_of == "Env" then
	-- 	return self.inherited_env:is_param_readonly(key)
	-- end
end

-- "inherits" from a different environment
--	this means that when trying to access an undefined parameter, it will check whether
--	the parameter exists in that environment, and if so, returns it
--	@Env env
--	@treturn nil
function Env:inherit(env)
	-- {{{ type checking
	assert(env.instance_of == "Env", "env should be an instance of Env. instead of " .. env.instance_of)
	-- }}}

	self.inherited_env = env
end

return Env
