-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

--- @classmod LanguageConstruct_ParameterExpansion
--	Describes a parameter expansion
local helpers = require("helpers")
local DatatypeValidator

local ParameterExpansion =
{
	instance_of = "LanguageConstruct_ParameterExpansion",
}

ParameterExpansion.__index = ParameterExpansion

function ParameterExpansion:new(t)
	-- {{{ type checking
	if t then
		assert(type(t) == "table", "t should be of type table. instead of " .. type(t))
		if t.parameter_name and type(t.parameter_name) ~= "string" then error("t.parameter_name should be of type string. instead of " .. type(t.parameter_name)) end
	end
	-- }}}

	t = t or {}

	self = setmetatable(
	{

		--- @string[opt=""] parameter_name name of the parameter to fetch
		parameter_name = t.parameter_name or "",

	}, ParameterExpansion)

	return self
end

--- evaluates the parameter expansion and returns the value
--	@ShellContext ctx
--	@treturn string the value of the parameter
--
-- {{{ evaluating
function ParameterExpansion:evaluate(ctx)
	DatatypeValidator = DatatypeValidator or require("datatype_validator")

	local parameter_value = ctx.env:get(self.parameter_name)

	return parameter_value
end
-- }}}

local parameter_identifier_set = helpers.concat_sets(helpers.special_parameter_chars, helpers.valid_identifier_chars_set, helpers.numeric_chars_set)

--- parses a parameter expansion
--	@ParserContext parser_ctx
--	@treturn nil
--
-- {{{ parsing
function ParameterExpansion:parse(parser_ctx)
	local char
	local idx
	local parameter_name = ""
	local modifier_arg
	char = parser_ctx:consume(2)

	if helpers.is_valid_identifier_char(char, true) or helpers.special_parameter_chars[char] then -- expected: $<var>
		parameter_name = parameter_name .. char

		-- state: $<first character>*

		idx = parser_ctx.character_index
		parser_ctx:skip_until(parameter_identifier_set, true)
		parameter_name = parameter_name .. parser_ctx.text:sub(idx, parser_ctx.character_index - 1)

		-- state: $<var>*

	elseif char == "{" then -- expected: ${<var>}
		char = parser_ctx:consume()

		-- state: ${*

		if not helpers.is_valid_identifier_char(char, true) then
			error("Invalid parameter name")
		end

		parameter_name = parameter_name .. char

		idx = parser_ctx.character_index
		parser_ctx:skip_until(parameter_identifier_set, true)
		parameter_name = parameter_name .. parser_ctx.text:sub(idx, parser_ctx.character_index - 1)

		-- state: ${<var>*

		parser_ctx:expect("}")

		-- state: ${var}*
	else
		error("Invalid parameter name")
	end

	if #parameter_name ~= 0 then
		self.parameter_name = parameter_name
	else
		error("parameter has no identifier")
	end
end
-- }}}

function ParameterExpansion:dump(dump_ctx)
	dump_ctx:write_text(string.format([[%s %s]], dump_ctx:color("(ParameterExpansion)", "ConstructSpecifier"), dump_ctx:color(self.parameter_name, "String")))
end

return ParameterExpansion
