-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

local helpers = require("helpers")
local Expression = require("language_constructs.expression")
local Shell = require("language_constructs.shell")
local ShellContext = require("shell_context")
local DatatypeValidator = require("datatype_validator")

local WhileLoop =
{
	instance_of = "LanguageConstruct_WhileLoop",
}

WhileLoop.__index = WhileLoop

function WhileLoop:new(t)
	t = t or {}
	self = setmetatable(
	{
		expression = t.expression or nil,
		subshell = t.subshell or nil
	}, WhileLoop)

	return self
end

local function evaluate_as_bool(value)
	if type(value) == "table" and value.instance_of == "TypedList" then
		return #value.container ~= 0
	end

	local datatype = DatatypeValidator.get_type_of(value)

	if type(value) == "table" and value.instance_of == "TypedList" then
		return #value > 0
	elseif DatatypeValidator.registered_types[datatype]
		and DatatypeValidator.registered_types[datatype].to_bool then
		return DatatypeValidator.registered_types[datatype].to_bool(value)
	else
		return false
	end
end

function WhileLoop:evaluate(ctx, max_frustration)
	-- {{{ type checking
	assert(ctx.instance_of == "ShellContext", "ctx should be an instance of ShellContext. instead of " .. ctx.instance_of)
	-- }}}

	max_frustration = max_frustration or 10000
	local frustration = 0
	local subshell_ctx = ShellContext:new()
	subshell_ctx.privilege_cache = ctx.privilege_cache

	ctx:interrupt()

	local function iterate()
		if evaluate_as_bool(self.expression:evaluate(ctx)) then
			assert(frustration <= max_frustration, "loop exceeded maximum iterations, as defined by lush_max_loop_iterations setting")

			self.subshell:evaluate(ctx,
			{
				subshell_context = subshell_ctx,
				complete_callback = function()
					return iterate()
				end
			})
			subshell_ctx.stdout:clear()

			frustration = frustration + 1
		else
			return ctx:stop_interruption()
		end
	end

	iterate()
end

local subshell_terminator = helpers.create_trie({"end"})
local whitespace_trie = helpers.create_trie({" ", "\t"})

function WhileLoop:parse(parser_ctx)
	-- {{{ type checking
	assert(parser_ctx.instance_of == "ParserContext", "parser_ctx should be an instance of ParserContext. instead of " .. parser_ctx.instance_of)
	-- }}}

	local success, _ = parser_ctx:match_fast("while", true)

	assert(success, "expected 'while'")

	parser_ctx:consume_whitespaces()

	self.expression = Expression:new()
	self.expression:parse(parser_ctx, whitespace_trie)

	parser_ctx:consume_whitespaces()

	local success, _ = parser_ctx:match_fast("do", true)

	assert(success, "expected 'do'")

	parser_ctx:consume_whitespaces()

	self.subshell = Shell:new()
	self.subshell:parse(parser_ctx, {}, subshell_terminator)

	parser_ctx:expect("e")
	parser_ctx:expect("n")
	parser_ctx:expect("d")
end

function WhileLoop:dump(dump_ctx)
	dump_ctx:write_text(dump_ctx:color("(WhileLoop)", "ConstructSpecifier"))
	dump_ctx:new_line()
	dump_ctx:write_text("[")
	dump_ctx:indent(1)
	dump_ctx:new_line()

	dump_ctx:write_text("Conditional Expression: ")
	self.expression:dump(dump_ctx)
	dump_ctx:new_line()

	self.subshell:dump(dump_ctx)

	dump_ctx:indent(-1)
	dump_ctx:new_line()
	dump_ctx:write_text("]")
end

return WhileLoop
