-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

-- @classmod range
--	a class that describes a range of numbers

local Range =
{
	instance_of = "Range"
}

Range.__index = Range

function Range:new(t)
	t = t or {}

	-- {{{ type checking
	if t.start and type(t.start) ~= "number" then error("t.start should be of type number. instead of " .. type(t.start)) end
	if t.finish and type(t.finish) ~= "number" then error("t.finish should be of type number. instead of " .. type(t.finish)) end
	if t.start_inclusive and type(t.start_inclusive) ~= "boolean" then error("t.start_inclusive should be of type boolean. instead of " .. type(t.start_inclusive)) end
	if t.finish_inclusive and type(t.finish_inclusive) ~= "boolean" then error("t.finish_inclusive should be of type boolean. instead of " .. type(t.finish_inclusive)) end
	-- }}}

	self = setmetatable(
	{
		start = t.start or -math.huge,

		finish = t.finish or math.huge,
	},
	Range)

	if t.start_inclusive == nil then
		self.start_inclusive = true
	else
		self.start_inclusive = t.start_inclusive
	end

	if t.finish_inclusive == nil then
		self.finish_inclusive = true
	else
		self.finish_inclusive = t.finish_inclusive
	end

	return self
end

function Range:__tostring()
	local output_string = ""

	if self.start_inclusive then
		output_string = output_string .. "<"
	else
		output_string = output_string .. "("
	end

	if self.start ~= math.huge then
		output_string = output_string .. tostring(self.start)
	end

	output_string = output_string .. ":"

	if self.finish ~= math.huge then
		output_string = output_string .. tostring(self.finish)
	end

	if self.finish_inclusive then
		output_string = output_string .. ">"
	else
		output_string = output_string .. ")"
	end

	return output_string
end

function Range:get_integer_boundries()
	local start = math.ceil(self.start)
	local finish = math.floor(self.finish)
	if not self.start_inclusive then
		start = start + 1
	end

	if not self.finish_inclusive then
		finish = finish - 1
	end

	return start, finish
end

-- checks whether the number is inside the range
--	@number number the number to check
--	@treturn bool
function Range:contains(number)
	if not ((self.start_inclusive and self.start <= number) or self.start < number) then
		return false
	end

	if not ((self.finish_inclusive and self.finish >= number) or self.finish > number) then
		return false
	end

	return true
end

-- returns whether the range is infinite
function Range:is_infinite()
	return self.start == math.huge or self.finish == math.huge
end

-- runs a function for every integer in the range, does nothing if the range is infinite
--	@func func the function to run, takes the Range instance as an argument and a number
--	@treturn nil
function Range:iterator()
	if self:is_infinite() then
		return function()
			return nil
		end
	end

	local number = math.ceil(self.start) - 1
	local finish = math.floor(self.finish)


	if not self.start_inclusive then
		-- if isnt inclusive and iteration start is equals to the boundary, icrement by one
		number = number + 1
	end

	if not self.finish_inclusive then
		-- if isnt inclusive and iteration start is equals to the boundary, decrement by one
		finish = finish - 1
	end

	return function()
		if number >= finish then
			return nil
		end

		number = number + 1
		return number
	end
end

return Range
