-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

require("tests.busted_definitions")
require("tests/command_definitions")

describe("LanguageConstruct_Shell class: ",
function()
	local Shell = require("language_constructs.shell")
	local ParserContext = require("parser_context")
	local ShellContext = require("shell_context")
	local helpers = require("helpers")
	local shell_instance
	local shell_context_instance
	local parser_context_instance

	before_each(
	function()
		shell_instance = Shell:new()
		shell_context_instance = ShellContext:new()
		shell_context_instance.env:inherit(lush.global_env)
		parser_context_instance = ParserContext:new()
	end)

	describe("parses correctly",
	function()

		it("when given a single command",
		function()
			parser_context_instance.text = "command arg1 arg2"

			shell_instance:parse(parser_context_instance, {}, {})

			assert.equals("arg1", shell_instance.actions[1].args[1]:evaluate(shell_context_instance))
			assert.equals("arg2", shell_instance.actions[1].args[2]:evaluate(shell_context_instance))

			assert.equals(18, parser_context_instance.character_index)
		end)

		it("when given sequential commands",
		function()
			parser_context_instance.text = [[command arg1 arg2; command2 arg1 arg2]]

			shell_instance:parse(parser_context_instance, {}, {})

			assert.equals("arg1", shell_instance.actions[1].args[1]:evaluate(shell_context_instance))
			assert.equals("arg2", shell_instance.actions[1].args[2]:evaluate(shell_context_instance))

			assert.equals(1, shell_instance.actions[2])

			assert.equals("arg1", shell_instance.actions[3].args[1]:evaluate(shell_context_instance))
			assert.equals("arg2", shell_instance.actions[3].args[2]:evaluate(shell_context_instance))

			assert.equals(38, parser_context_instance.character_index)
		end)

		it("when given pipes",
		function()
			parser_context_instance.text = [[command arg1 arg2 | command2 arg1 arg2]]

			shell_instance:parse(parser_context_instance, {}, {})

			assert.equals("arg1", shell_instance.actions[1].args[1]:evaluate(shell_context_instance))
			assert.equals("arg2", shell_instance.actions[1].args[2]:evaluate(shell_context_instance))

			assert.equals(2, shell_instance.actions[2])

			assert.equals("arg1", shell_instance.actions[3].args[1]:evaluate(shell_context_instance))
			assert.equals("arg2", shell_instance.actions[3].args[2]:evaluate(shell_context_instance))

			assert.equals(39, parser_context_instance.character_index)
		end)

		it("when given ANDs",
		function()
			parser_context_instance.text = [[command arg1 arg2 && command2 arg1 arg2]]

			shell_instance:parse(parser_context_instance, {}, {})

			assert.equals("arg1", shell_instance.actions[1].args[1]:evaluate(shell_context_instance))
			assert.equals("arg2", shell_instance.actions[1].args[2]:evaluate(shell_context_instance))

			assert.equals(3, shell_instance.actions[2])

			assert.equals("arg1", shell_instance.actions[3].args[1]:evaluate(shell_context_instance))
			assert.equals("arg2", shell_instance.actions[3].args[2]:evaluate(shell_context_instance))

			assert.equals(40, parser_context_instance.character_index)
		end)

		it("when given ORs",
		function()
			parser_context_instance.text = [[command arg1 arg2 || command2 arg1 arg2]]

			shell_instance:parse(parser_context_instance, {}, {})

			assert.equals("arg1", shell_instance.actions[1].args[1]:evaluate(shell_context_instance))
			assert.equals("arg2", shell_instance.actions[1].args[2]:evaluate(shell_context_instance))

			assert.equals(4, shell_instance.actions[2])

			assert.equals("arg1", shell_instance.actions[3].args[1]:evaluate(shell_context_instance))
			assert.equals("arg2", shell_instance.actions[3].args[2]:evaluate(shell_context_instance))

			assert.equals(40, parser_context_instance.character_index)
		end)

		it("when command ends with ;",
		function()
			parser_context_instance.text = [[command arg1 arg2;]]

			shell_instance:parse(parser_context_instance, {}, {})

			assert.equals("arg1", shell_instance.actions[1].args[1]:evaluate(shell_context_instance))
			assert.equals("arg2", shell_instance.actions[1].args[2]:evaluate(shell_context_instance))

			assert.equals(1, shell_instance.actions[2])

			assert.equals(19, parser_context_instance.character_index)
		end)

		it("when subshell is followed by a command",
		function()
			parser_context_instance.text = [[(command one; command two); command three]]

			shell_instance:parse(parser_context_instance, {}, {})

			assert.equals("one", shell_instance.actions[1].actions[1].args[1]:evaluate(shell_context_instance))
			assert.equals("two", shell_instance.actions[1].actions[3].args[1]:evaluate(shell_context_instance))
			assert.equals("three", shell_instance.actions[3].args[1]:evaluate(shell_context_instance))

			assert.equals(1, shell_instance.actions[1].actions[2])
			assert.equals(1, shell_instance.actions[2])

			assert.equals(42, parser_context_instance.character_index)
		end)
	end)

	describe("evaluates correctly: ",
	function()
		-- for these tests im also parsing instead of just evaluating
		-- like in other test files for language constructs
		-- this is because manually setting all the members of a  shell is too long

		it("simply printing to STDOUT",
		function()
			parser_context_instance.text = [[echo string]]

			shell_instance:parse(parser_context_instance, {}, {})

			shell_instance:evaluate(shell_context_instance, {complete_callback = function() end})

			assert.equals("string", shell_context_instance.stdout:get(1))
		end)

		it("piping line",
		function()
			parser_context_instance.text = [[echo string | append " another string"]]

			shell_instance:parse(parser_context_instance, {}, {})

			shell_instance:evaluate(shell_context_instance, {complete_callback = function() end})

			assert.equals("string another string", shell_context_instance.stdout:get(1))
		end)

		it("success operator",
		function()
			parser_context_instance.text = [[echo text1 && echo text2]]

			shell_instance:parse(parser_context_instance, {}, {})

			shell_instance:evaluate(shell_context_instance, {complete_callback = function() end})

			assert.equals("text1", shell_context_instance.stdout:get(1))
			assert.equals("text2", shell_context_instance.stdout:get(2))
		end)

		it("failure operator",
		function()
			parser_context_instance.text = [[echo text1 || echo text2]]

			shell_instance:parse(parser_context_instance, {}, {})

			shell_instance:evaluate(shell_context_instance, {complete_callback = function() end})

			assert.equals("text1", shell_context_instance.stdout:get(1))
			assert.equals(nil, shell_context_instance.stdout:get(2))
		end)

		it("sequential operator",
		function()
			parser_context_instance.text = [[echo text1; echo text2]]

			shell_instance:parse(parser_context_instance, {}, {})

			shell_instance:evaluate(shell_context_instance, {complete_callback = function() end})

			assert.equals("text1", shell_context_instance.stdout:get(1))
			assert.equals("text2", shell_context_instance.stdout:get(2))
		end)

		-- it("cant set global variables unless they have been previously defined",
		-- function()
		-- 	parser_context_instance.text = [[echo value | set parameter]]
		--
		-- 	shell_instance:parse(parser_context_instance, {}, {})
		--
		-- 	shell_instance:evaluate(shell_context_instance)
		--
		-- 	assert.equals(nil, shell_context_instance.env:get("parameter"))
		-- end)
		--
		-- it("can set local parameters",
		-- function()
		-- 	parser_context_instance.text = [[echo value | set parameter; echo $parameter]]
		--
		-- 	shell_instance:parse(parser_context_instance, {}, {})
		--
		-- 	shell_instance:evaluate(shell_context_instance)
		--
		-- 	assert.equals("value", shell_context_instance.stdout:get(1))
		-- 	assert.equals(nil, shell_context_instance.env:get("parameter"))
		-- end)

		it("sub shells work",
		function()
			parser_context_instance.text = [[(echo text)]]

			shell_instance:parse(parser_context_instance, {}, {})

			shell_instance:evaluate(shell_context_instance,
			{
				share_environment = true,
				complete_callback = function()
					assert.equals("text", shell_context_instance.stdout:get(1))
				end
			})

		end)

		it("STDOUT of commands without pipes is the STDOUT of the subshell",
		function()
			parser_context_instance.text = [[(echo text1; echo text2) | append end]]

			shell_instance:parse(parser_context_instance, {}, {})

			shell_instance:evaluate(shell_context_instance,
			{
				share_environmen = true,
				complete_callback = function()
					assert.equals("text1end", shell_context_instance.stdout:get(1))
					assert.equals("text2end", shell_context_instance.stdout:get(2))
				end
			})
		end)

		it("comments are ignored",
		function()
			parser_context_instance.text = [[echo hello // this is a comment]]

			shell_instance:parse(parser_context_instance, {}, {})

			shell_instance:evaluate(shell_context_instance,
			{
				share_environmen = true,
				complete_callback = function()
					assert.equals("hello", shell_context_instance.stdout:get(1))
					assert.equals(nil, shell_context_instance.stdout:get(2))
				end
			})
		end)

		it("comments are ignored when they're on their own line",
		function()
			parser_context_instance.text = [[echo hello
			// this is a comment
			echo world]]

			shell_instance:parse(parser_context_instance, {}, {})

			shell_instance:evaluate(shell_context_instance,
			{
				share_environmen = true,
				complete_callback = function()
					assert.equals("hello", shell_context_instance.stdout:get(1))
					assert.equals("world", shell_context_instance.stdout:get(2))
				end
			})
		end)

		it("comments are ignored when they're on the last line",
		function()
			parser_context_instance.text = [[echo hello
			// this is a comment]]

			shell_instance:parse(parser_context_instance, {}, {})

			shell_instance:evaluate(shell_context_instance,
			{
				share_environmen = true,
				complete_callback = function()
					assert.equals("hello", shell_context_instance.stdout:get(1))
				end
			})
		end)

		it("more than one comments in a row are ignored",
		function()
			parser_context_instance.text = [[echo hello
			// this is a comment
			// this is an another comment
			// this is a yet another comment
			]]

			shell_instance:parse(parser_context_instance, {}, {})

			shell_instance:evaluate(shell_context_instance,
			{
				share_environmen = true,
				complete_callback = function(success)
					assert.equals(true, success)
					assert.equals("hello", shell_context_instance.stdout:get(1))
				end
			})
		end)
	end)

end)
