-- Copyright (C) 2024 rstcxk
--
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
--
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.

lush.Command:new(
{
	name = "pow",
	description = "returns the first argument to the power of second argument",
	arguments =
	{
		{
			name = "number",
			type = "number",
			single = true,
		},
		{
			name = "power",
			type = "number",
			single = true,
		},
	},
	callback = function(ctx, args, options)
		ctx.stdout:push(args.number ^ args.power)
	end,
})

lush.Command:new(
{
	name = "sqrt",
	description = "returns number squared",
	arguments =
	{
		{
			name = "number",
			type = "number",
			single = true,
		},
	},
	callback = function(ctx, args, options)
		ctx.stdout:push(math.sqrt(args.number))
	end,
})

lush.Command:new(
{
	name = "abs",
	description = "returns the absolute value of given number",
	arguments =
	{
		{
			name = "number",
			type = "number",
			single = true,
		},
	},
	callback = function(ctx, args, options)
		ctx.stdout:push(math.abs(args.number))
	end,
})

lush.Command:new(
{
	name = "floor",
	description = "returns largest integer smaller or equal to given number",
	arguments =
	{
		{
			name = "number",
			type = "number",
			single = true,
		},
	},
	callback = function(ctx, args, options)
		ctx.stdout:push(math.floor(args.number))
	end,
})

lush.Command:new(
{
	name = "ceil",
	description = "returns smallest integer larger or equal to given number",
	arguments =
	{
		{
			name = "number",
			type = "number",
			single = true,
		},
	},
	callback = function(ctx, args, options)
		ctx.stdout:push(math.ceil(args.number))
	end,
})

lush.Command:new(
{
	name = "round",
	description = "rounds the given number",
	arguments =
	{
		{
			name = "number",
			type = "number",
			single = true,
		},
	},
	callback = function(ctx, args, options)
		ctx.stdout:push(math.round(args.number))
	end,
})

lush.Command:new(
{
	name = "min",
	description = "rounds the smallest of the given numbers",
	arguments =
	{
		{
			rule_type = "block",
			name = "numbers",
			type = "number",
			required = true
		},
	},
	callback = function(ctx, args, options)
		local min = math.huge
		for _, v in pairs(args.numbers) do
			if v < min then
				min = v
			end
		end

		ctx.stdout:push(min)
	end,
})

lush.Command:new(
{
	name = "max",
	description = "rounds the largest of the given numbers",
	arguments =
	{
		{
			rule_type = "block",
			name = "numbers",
			type = "number",
			required = true
		},
	},
	callback = function(ctx, args, options)
		local max = -math.huge
		for _, v in pairs(args.numbers) do
			if v > max then
				max = v
			end
		end

		ctx.stdout:push(max)
	end,
})

lush.Command:new(
{
	name = "clamp",
	description = "clamp a value to a given range",
	arguments =
	{
		{
			name = "number",
			type = "number",
			single = true
		},
		{
			name = "range",
			type = "range",
			single = true
		},
	},
	callback = function(ctx, args, options)
		local start, finish = args.range:get_integer_boundries()
		ctx.stdout:push(math.max(math.min(args.number, finish), start))
	end,
})

lush.Command:new(
{
	name = "random",
	description = "returns a random number in the provided range, or between 1 or 0 if no range is given",
	arguments =
	{
		{
			name = "range",
			type = "range",
			single = true,
			optional = true
		}
	},
	callback = function(ctx, args, options)
		if args.range then
			local start, finish = args.range:get_integer_boundries()
			ctx.stdout:push(math.random(start, finish))
		else
			ctx.stdout:push(math.random())
		end
	end,
})

local transformations =
{
 	identity                  = {x = {x = 1,  y = 0,  z = 0},  y = {x = 0,  y  = 1,  z = 0}, z = {x = 0,  y  = 0, z = 1 }},

 	rotate_z_counterclockwise = {x = {x = 0, y  = 1,  z = 0 }, y = {x = -1, y  = 0,  z = 0}, z = {x = 0,   y = 0, z = 1 }},
 	rotate_z_clockwise        = {x = {x = 0, y  = -1, z = 0 }, y = {x = 1,   y = 0,  z = 0}, z = {x = 0,  y  = 0, z = 1 }},
 	rotate_z_180              = {x = {x = -1, y = 0,  z = 0 }, y = {x = 0,  y  = -1, z = 0}, z = {x = 0,   y = 0, z = 1 }},
	flip_z                    = {x = {x = 1,  y = 0,  z = 0 }, y = {x = 0, y   = 1,  z = 0}, z = {x = 0,  y  = 0, z = -1}},

	rotate_y_counterclockwise = {x = {x = 0, y  = 0,  z = 1 }, y = {x = 0,  y  = 1,  z = 0}, z = {x = -1, y  = 0, z = 0 }},
 	rotate_y_clockwise        = {x = {x = 0, y  = 0,  z = -1}, y = {x = 0,   y = 1,  z = 0}, z = {x = 1,  y  = 0, z = 0 }},
	rotate_y_180              = {x = {x = -1, y = 0,  z = 0 }, y = {x = 0,  y  = 1,  z = 0}, z = {x = 0,  y  = 0, z = -1}},
 	flip_y                    = {x = {x = 1,  y = 0,  z = 0 }, y = {x = 0, y   = -1, z = 0}, z = {x = 0,   y = 0, z = 1 }},

 	rotate_x_counterclockwise = {x = {x = 1, y  = 0,  z = 0 }, y = {x = -1, y  = 0,  z = 0}, z = {x = 1,   y = 0, z = 0 }},
	rotate_x_clockwise        = {x = {x = 1, y  = 0,  z = 0 }, y = {x = -1, y  = 0,  z = 0}, z = {x = -1,  y = 0, z = 0 }},
	rotate_x_180              = {x = {x = 1, y  = 0,  z = 0 }, y = {x = 0,  y  = -1, z = 0}, z = {x = 0,   y = 0, z = -1}},
 	flip_x                    = {x = {x = -1, y = 0,  z = 0 }, y = {x = 0,  y  = 1,  z = 0}, z = {x = 0,  y  = 0, z = 1 }},
}

lush.global_env:set("transformations", transformations, true, true)

lush.Command:new(
{
	name = "transform",
	description = "performs a linear transformation on vectors",
	options = {["origin"] = true},
	arguments =
	{
		{
			name = "vectors",
			type = "vector",
			single = false
		},
		{
			rule_type = "block",
			name = "transformations",
			type = "table",
			required = true,
		},
	},
	callback = function(ctx, args, options)
		local origin = options.origin and lush.DatatypeValidator.cast_to(options["origin"], "vector") or vector.new(0, 0, 0)
		assert(origin, "Option `origin` is invalid")
		-- for this, i assume the transformation to have this format:
		-- {
		--  x = <vector>,
		--  y = <vector>,
		--  z = <vector>
		-- }
		-- The outermost table has the xyz members, which are the identity vectors

		-- this function changes the original vector, it dosen't return a new one
		local function transform(vec, transform)
			local tmp_x = vec.x * transform.x.x + vec.y * transform.y.x + vec.z * transform.z.x
			local tmp_y = vec.x * transform.x.y + vec.y * transform.y.y + vec.z * transform.z.y
			local tmp_z = vec.x * transform.x.z + vec.y * transform.y.z + vec.z * transform.z.z

			vec.x = tmp_x
			vec.y = tmp_y
			vec.z = tmp_z
		end

		local transformation = {x = vector.new(1, 0, 0), y = vector.new(0, 1, 0), z = vector.new(0, 0, 1)}

		for _, trans in args.transformations:iterator() do
			transform(transformation.x, trans)
			transform(transformation.y, trans)
			transform(transformation.z, trans)
		end

		origin = origin or vector.new(0, 0, 0)
		for _, pos in args.vectors:iterator() do
			local output_vector = vector.add(origin, pos)

			transform(output_vector, transformation)

			ctx.stdout:push(output_vector)
		end
	end,
})
