-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

-- @classmod LanguageConstruct_CommandInvocation
--	Describes a command invocation

local helpers = require("helpers")

local Expression

local CommandInvocation =
{
	instance_of = "LanguageConstruct_CommandInvocation",
}

CommandInvocation.__index = CommandInvocation

function CommandInvocation:new(t)
	if t == nil then
		t = {}
	end

	-- {{{ type checking
	assert(type(t) == "table", "t should be of type table. instead of " .. type(t))
	if t.args and type(t.args) ~= "table" then error("t.args should be of type table. instead of " .. type(t.args)) end
	-- }}}

	self = setmetatable(
	{
		command = t.command or nil,
		-- name = t.name or nil, --i dont want to require the Expression module here, because i doubt it would ever be useful
		args = t.args or {}
	}, CommandInvocation)

	return self
end

function CommandInvocation:evaluate(ctx)
	-- {{{ type checking
	assert(ctx.instance_of == "ShellContext", "ctx should be an instance of ShellContext. instead of " .. ctx.instance_of)
	-- }}}

	local command = self.command:evaluate(ctx)
	local command_name

	if type(command) == "string" then
		command_name = command
		command = ctx.env:get(command)
	end

	if type(command) ~= "function" and not (type(command) == "table" and command.__call) then
		if type(command) == "nil" then
			if command_name then
				error("command not found: " .. command_name)
			else
				error("command not found")
			end
		else
			error("trying to call an invalid command")
		end
	end

	local args = {}

	for _, argument in ipairs(self.args) do
		table.insert(args, argument:evaluate(ctx))
	end

	command(ctx, args)
end

local white_chars_trie = helpers.create_trie({" ", "\t"})
local argument_terminator_cache = {}

function CommandInvocation:parse(parser_ctx, terminators)
	-- {{{ type checking
	assert(parser_ctx.instance_of == "ParserContext", "parser_ctx should be an instance of ParserContext. instead of " .. parser_ctx.instance_of)
	-- }}}

	Expression = Expression or require("language_constructs.expression")

	local argument_terminators

	if argument_terminator_cache[terminators] then
		argument_terminators = argument_terminator_cache[terminators]
	else
		local tmp = helpers.add_tries(white_chars_trie, terminators)
		argument_terminator_cache[terminators] = tmp
		argument_terminators = tmp
	end

	local arg
	local matching, _

	if not self.command then
		-- partial parsing
		self.command = Expression:new()
		self.command:parse(parser_ctx, argument_terminators)
	end

	-- state: <command_name>*

	while true do
		parser_ctx:consume_whitespaces()
		matching, _ = parser_ctx:match(argument_terminators)

		-- state: <command_name><arguments if any><whitespace>*

		if matching or parser_ctx:is_EOF() then
			-- state: <command_name><arguments>*<control operator>
			break
		end

		arg = Expression:new()
		table.insert(self.args, arg)
		arg:parse(parser_ctx, argument_terminators)

		-- state: <command_name><arguments if any><whitespace><argument>*
	end

end

function CommandInvocation:dump(dump_ctx)
	dump_ctx:write_text(dump_ctx:color("(CommandInvocation)", "ConstructSpecifier"))
	dump_ctx:new_line()
	dump_ctx:write_text("[")
	dump_ctx:indent(1)
	dump_ctx:new_line()

	dump_ctx:write_text("Command: ", true)
	self.command:dump(dump_ctx)
	dump_ctx:new_line()

	dump_ctx:write_text("Arguments: ")
	dump_ctx:new_line()
	dump_ctx:write_text("[")
	dump_ctx:indent(1)

	for _, v in pairs(self.args) do
		dump_ctx:new_line()
		v:dump(dump_ctx)
	end

	dump_ctx:indent(-1)
	dump_ctx:new_line()
	dump_ctx:write_text("]")

	dump_ctx:indent(-1)
	dump_ctx:new_line()
	dump_ctx:write_text("]")
end

return CommandInvocation
