-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

--- @classmod LanguageConstruct_VectorExpression
--	describes a vector

local helpers = require("helpers")
local Expression
local NumberLiteral
local DatatypeValidator = require("datatype_validator")

local VectorExpression =
{
	instance_of = "LanguageConstruct_VectorExpression",
}

VectorExpression.__index = VectorExpression

function VectorExpression:new(t)
	t = t or {}
	self = setmetatable(
	{
		members = t.members or {0, 0, 0},
		relative = t.members or {false, false, false},
		caret_notation = t.caret_notation or false
	}, VectorExpression)

	return self
end

function VectorExpression:evaluate(ctx)
	-- {{{ type checking
	assert(ctx.instance_of == "ShellContext", "ctx should be an instance of ShellContext. instead of " .. ctx.instance_of)
	-- }}}
	local output = {}
	for idx = 1, 3 do
		if self.caret_notation then
			-- @todo implement caret notation
		end

		local value = self.members[idx]:evaluate(ctx)

		if type(value) ~= "number" then
			local value, success = DatatypeValidator.cast_to(value, "number", ctx)

			assert(success, "cannot convert members of a vector into numbers")
		end

		-- if is relative to player's position
		if self.relative[idx] then
			table.insert(output, value + ctx.env:get("pos")[idx])
		else
			table.insert(output, value)
		end
	end

	return output
end

local subexpression_terminators      = helpers.create_trie({ ",", " "})
local last_subexpression_terminators = helpers.create_trie({")"})

function VectorExpression:parse(parser_ctx)
	-- {{{ type checking
	assert(parser_ctx.instance_of == "ParserContext", "parser_ctx should be an instance of ParserContext. instead of " .. parser_ctx.instance_of)
	-- }}}

	NumberLiteral = NumberLiteral or require("language_constructs.number_literal")
	Expression = Expression or require("language_constructs.expression")

	local char = parser_ctx:consume()
	local terminators = subexpression_terminators

	if char == "^" then
		self.caret_notation = true

		parser_ctx:expect("(")
	elseif char ~= "(" then
		error("expected \"(\", instead of " .. char)
	end

	for idx = 1, 3 do
		-- state: *
		parser_ctx:consume_whitespaces()

		char = parser_ctx:peek()

		if char == "~" then
			assert(not self.caret_notation, "caret notation cant be combined with relative vectors")

			self.relative[idx] = true
			parser_ctx:advance()
			-- state: ~*
		end

		-- state: <tilde?>*

		if idx == 3 then
			-- for the third iteration, add ) to the terminators
			terminators = last_subexpression_terminators
		end

		parser_ctx:consume_whitespaces()

		local matched, _ = parser_ctx:match(terminators, true)

		parser_ctx:consume_whitespaces()

		if matched then
			-- the number expression is missing, defaulting to 0
			self.members[idx] = NumberLiteral:new({value = 0})
		else
			self.members[idx] = Expression:new()
			self.members[idx]:parse(parser_ctx, terminators)
		end

		-- state: <tilde?><expression?>*

		assert(self.members[idx], "invalid vector memeber")

		parser_ctx:consume_whitespaces()

		matched, _ = parser_ctx:match(terminators, true)

		assert(matched, idx == 3 and "vector must be terminated with an \")\"" or "vector members need to be seperated with a comma")
		-- state: <tilde?><expression><peroids and spaces>*
	end
end

local axis_names = {"X", "Y", "Z"}
function VectorExpression:dump(dump_ctx)
	dump_ctx:write_text(dump_ctx:color("(VectorExpression)", "ConstructSpecifier"))
	dump_ctx:new_line()
	dump_ctx:write_text("[")
	dump_ctx:indent(1)
	dump_ctx:new_line()

	for i = 1, 3 do
		dump_ctx:new_line()
		dump_ctx:write_text(string.format("%s: %s", axis_names[i], self.relative[i] and "(relative)" or ""))
		self.members[i]:dump(dump_ctx)
	end

	dump_ctx:indent(-1)
	dump_ctx:new_line()
	dump_ctx:write_text("]")
end

return VectorExpression
