-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

---@diagnostic disable: duplicate-set-field

-- @submodule Command

local Command, RegisteredCommands = require("command")
local Interpreter = require("interpreter")
local ChatOutput = require("minetest.chat_output")
local helpers = require("helpers")
local ShellContext = require("shell_context")
local DatatypeValidator = require("datatype_validator")

local old_constructor = Command.new
Command.new = function(self, t)
	self = old_constructor(self, t)

	-- {{{ type checking
	if t.privs and type(t.privs) ~= "table" then error("t.privs should be of type table. instead of " .. type(t.privs)) end
	if t.params and type(t.params) ~= "string" then error("t.params should be of type string. instead of " .. type(t.params)) end
	if t.description and type(t.description) ~= "string" then error("t.description should be of type string. instead of " .. type(t.description)) end
	-- }}}

	self.privs = t.privs or {}
	self.params = t.params or ""
	self.description = t.description or {}
	self.mod_origin = core.get_current_modname()

	if self.arguments and self.params == "" then
		self.params = DatatypeValidator.generate_param_hint(self.arguments, self.options)
	end

	if core.registered_chatcommands[self.name] then
		core.log("info", string.format("%s overriden by lush", self.name))
		core.unregister_chatcommand(self.name)
	end

	self:register_compatibility_command()

	return self
end

local old_unregister = Command.unregister
Command.unregister = function(self)
	old_unregister(self)
	core.unregister_chatcommand(self.name)
end

local old_call = Command.call
Command.call = function(self, ctx, args)
	local has_privs = true
	for k, v in pairs(self.privs) do
		if v ~= ctx.privilege_cache[k] then
			has_privs = false
			break
		end
	end

	if has_privs then
		old_call(self, ctx, args)
	else
		local _, missing = core.check_player_privs("singleplayer", self.privs)
		error("player is missing privilages: " .. table.concat(missing, ", "))
	end
end

Command.register_compatibility_command = function(self)
	core.register_chatcommand(self.name,
	{
		privs = self.privs,
		params = self.params,
		description = self.description,
		func = function(name, params)
			local ctx = ShellContext:new({env = lush.player_environments[name]})
			local success, message = Interpreter.run_command(self.name .. " " .. params, ctx)

			if success then
				ctx.stdout:cast_to("string")
				return true, table.concat(ctx.stdout.container, "\n")
			else
				return false, message
			end
		end
	})
end

core.register_on_chat_message(
function(name, message)
	if string.sub(message, 1, 1) == "!" then
		core.log("action", name .. " invoked a lush shell: \n" .. string.sub(message, 2))
		core.chat_send_player(name, core.colorize("orange", "invoked shell:"))
		local ctx = ShellContext:new({env = lush.player_environments[name], stdout = ChatOutput:new(name)})

		local shell_options =
		{
			share_environment = true,
			complete_callback = function(success, error_msg, ctx, parse_time, eval_time)
				if not success then
					core.chat_send_player(name, error_msg or "no error message provided")
				else
					local pretty_parse_time = helpers.pretty_time(parse_time)
					local pretty_eval_time = helpers.pretty_time(eval_time)
					core.chat_send_player(name, core.colorize("orange",
						string.format("(%s, %s)", pretty_parse_time, pretty_eval_time)))

					core.log("action", "exit code: " .. tostring(ctx.exit_status)
					.. "\nparsing time: " .. pretty_parse_time
					.. "\nevaluation time: " .. pretty_eval_time)
				end
			end
		}

		Interpreter.run_shell(string.sub(message, 2), ctx, shell_options)

		return true
	end
end)

-- core.register_on_mods_loaded(function()
-- 	for command, def in pairs(RegisteredCommands) do
-- 		def:register_compatibility_command()
-- 		core.registered_chatcommands[command].mod_origin = def.mod_origin
-- 	end
-- end)

return Command, RegisteredCommands
