-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

-- @classmod TypedList
-- this is a container type that can store either of the the allowed datatypes, but only one at a time

-- @todo actually make the tests pass again before getting distracted again

local DatatypeValidator = require("datatype_validator")

local TypedList =
{
	instance_of = "TypedList"
}

TypedList.__index = TypedList

-- constructor
--  @tab t table with matching keys as the class, with initialization values for the members
--  @treturn TypedList|nil the newly created Command instance
function TypedList:new(t)
	if not t then
		t = {}
	end

	-- {{{ type checking
	if t and type(t) ~= "table" then error("t should be of type table. instead of " .. type(t)) end
	if t.container and type(t.container) ~= "table" then error("t.container should be of type table. instead of " .. type(t.container)) end
	if t.type and type(t.type) ~= "string" then error("t.type should be of type string. instead of " .. type(t.type)) end
	-- }}}

	self = setmetatable(
	{
		-- @tab[opt={}] container table storing all the elements
		container = t.container or {},

		-- @string[opt=""] type specifies what datatype is currently allowed in the list <br>
		--	possible values are: <br>
		--	"String" <br>
		--	"Vector" <br>
		--	"ObjectRef" <br>
		--	"ItemStackRef" <br>
		--	and an empty string, which means that the list doesnt have a datatype assigned yet
		type = t.type or ""
	}, TypedList)

	return self
end

function TypedList:__len()
	return #self.container
end

-- adds a new element to the end of the list
--	@param value element to add
--	@treturn nil
function TypedList:push(value, ctx)
	-- {{{ type checking
	-- if not DatatypeValidator.is_valid_type(value) then error("trying to append an invalid datatype") end
	-- }}}
	-- trying to convert
	if self.type ~= "" and DatatypeValidator.get_type_of(value) ~= self.type then
		local converted_value, success = DatatypeValidator.cast_to(value, self.type, ctx)

		if success then
			value = converted_value
		else
			error(string.format("trying to append an invalid datatype, expected %s but got %s", self.type, DatatypeValidator.get_type_of(value)))
		end
	end

	-- if the table is empty and a type isnt yet specified
	if self.type == "" then
		self.type = DatatypeValidator.get_type_of(value)
	end

	table.insert(self.container, value)
end

-- removes the last element
--	@return the last element
function TypedList:pop()

	if #self.container == 1 then
		self.type = ""
	end

	return table.remove(self.container)
end

-- get element at the given index
--	@number index
--	@return element
function TypedList:get(index)
	-- {{{ type checking
		if type(index) ~= "number" then error("index should be of type number. instead of " .. type(index)) end
	-- }}}

	return self.container[index]
end

-- set element at the given index
--	@number index
--	@param value
--	@treturn nil
function TypedList:set(index, value, ctx)
	-- {{{ type checking
	if type(index) ~= "number" then error("index should be of type number. instead of " .. type(index)) end
	-- if not DatatypeValidator.is_valid_type(value) then error("trying to set an element with an invalid datatype") end
	-- }}}

	-- trying to convert
	if self.type ~= "" and DatatypeValidator.get_type_of(value) ~= self.type then
		local converted_value, success = DatatypeValidator.cast_to(value, self.type, ctx)

		if success then
			value = converted_value
		else
			error(string.format("trying to append an invalid datatype, expected %s but got %s", self.type, DatatypeValidator.get_type_of(value)))
		end
	end

	self.container[index] = value
end

-- clears the container
--	@treturn nil
function TypedList:clear()
	if #self.container ~= 0 then
		self.type = ""
		for i = 1, #self.container do
			self.container[i] = nil
		end
	end
end

function TypedList:concat(t, ctx)
	-- {{{ type checking
	if type(t) ~= "table" then error("t should be of type table. instead of " .. type(t)) end
	-- }}}
	local type
	if t.instance_of == "TypedList" then
		type = t.type
		t = t.container
	else
		type = DatatypeValidator.get_type_of(t[1])
	end

	if #t == 0 then
		return
	end

	if self.type ~= "" and type ~= self.type then
		local converted_values, success = DatatypeValidator.cast_list_to(t, self.type, ctx)

		if success then
			t = converted_values
		else
			error(string.format("trying to append an invalid datatype, expected %s but got %s", self.type, type))
		end
	elseif self.type == "" then
		self.type = type
	end

	for _, v in ipairs(t) do
		table.insert(self.container, v)
	end

end

function TypedList:cast_to(datatype, ctx)
	local success, converted_container
	converted_container, success = DatatypeValidator.cast_list_to(self.container, datatype, ctx)

	if success then
		self.container = converted_container
	end

	return success
end

function TypedList:printout()
	for _, v in pairs(self.container) do
		core.chat_send_all(tostring(v))
	end
end

function TypedList:iterator()
	local idx = 0

	return function()
		idx = idx + 1
		if idx > #self.container then return nil end
		return idx, self.container[idx]
	end
end

return TypedList
