-- Copyright (C) 2025 rstcxk
--
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
--
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.

lush.Command:new(
{
	name = "nset",
	description = "sets a node",
	privs = {give = true},
	arguments =
	{
		{
			name = "pos",
			type = "vector",
			single = false
		},
		{
			name = "node",
			type = "node_name",
			single = true
		},
	},
	callback = function(ctx, args, options)
		for _, pos in args.pos:iterator() do
			core.set_node(pos, {name = args.node})
		end
	end,
})

lush.Command:new(
{
	name = "ndel",
	description = "deletes a node",
	privs = {server = true},
	arguments =
	{
		{
			name = "pos",
			type = "vector",
			single = false
		},
	},
	callback = function(ctx, args, options)
		for _, pos in args.pos:iterator() do
			core.set_node(pos, {name = "air"})
		end
	end,
})

lush.Command:new(
{
	name = "nname",
	description = "prints the technical name of a node",
	arguments =
	{
		{
			name = "node_name",
			type = "node_name",
			single = true
		},
	},
	callback = function(ctx, args, options)
			ctx.stdout:push(args.node_name)
	end,
})

lush.Command:new(
{
	name = "ninfo",
	description = "prints information about the node at a given position",
	privs = {debug = true},
	arguments =
	{
		{
			rule_type = "terminal",
			name = "type",
			value =
			{
				["metadata"] = true,
				["inventory"] = true,
				["groups"] = true,
				["name"] = true,
				["param1"] = true,
				["param2"] = true,
			}
		},
		{
			name = "pos",
			type = "vector",
			single = true
		}
	},
	callback = function(ctx, args)

		if args.type == "metadata" then
			ctx.stdout:push(core.get_meta(args.pos):to_table())
		elseif args.type == "inventory" then
			local inv = core.get_inventory({type = "node", pos = args.pos})
			local indexes = {}

			for list_name, list in pairs(inv:get_lists()) do
				indexes[list_name] = {}
				for i, _ in pairs(list) do
					table.insert(indexes[list_name], i)
				end
			end

			ctx.stdout:push(lush.InventoryItemList:new(
			{
				location = {type = "node", pos = args.pos},
				indexes = indexes
			}))
		elseif args.type == "groups" then
			local nodedef = core.registered_nodes[core.get_node(args.pos).name]

			for k, v in pairs(nodedef.groups) do
				ctx.stdout:push(tostring(k) .. "=" .. tostring(v))
			end
		elseif args.type == "name" then
			ctx.stdout:push(core.get_node(args.pos).name)
		elseif args.type == "param1" then
			ctx.stdout:push(core.get_node(args.pos).param1)
		elseif args.type == "param2" then
			ctx.stdout:push(core.get_node(args.pos).param2)
		end
	end,
})

lush.Command:new(
{
	name = "ngroup",
	description = "returns a bool indicating whether the node at a given position belongs to a certain group",
	privs = {server = true},
	arguments =
	{
		{
			name = "node",
			type = "node_name",
			single = true
		},
		{
			name = "groupname",
			type = "string",
			single = true
		}
	},
	callback = function(ctx, args)
		ctx.stdout:push((core.get_item_group(args.node, args.groupname) or 0) > 0)
	end
})

lush.Command:new(
{
	name = "nsetinfo",
	description = "sets properties of nodes",
	privs = {debug = true, give = true},
	arguments =
	{
		{
			rule_type = "terminal",
			name = "type",
			value =
			{
				["metadata"] = true,
				["inventory"] = true,
				["groups"] = true,
				["name"] = true,
				["param1"] = true,
				["param2"] = true,
			}
		},
		{
			name = "pos",
			type = "vector",
		},
		{
			name = "value",
			single = true
		}
	},
	callback = function(ctx, args, options)
		local success

		if args.type == "metadata" then
			args.value, success = lush.DatatypeValidator.cast_to(args.value, "table")

			if not success then
				error("invalid argument #3: invalid type")
			end

			for _, pos in args.pos:iterator() do
				core.get_meta(pos):from_table(args.value)
			end
		elseif args.type == "inventory" then
			args.value, success = lush.DatatypeValidator.cast_to(args.value, "inventory_item_list")

			if not success then
				error("invalid argument #3: invalid type")
			end

			for _, pos in args.pos:iterator() do
				local inv = core.get_inventory({type = "node", pos = pos})
				local inv_list = inv:get_lists()

				for listname, list in pairs(args.value:get_lists()) do
					for index, stack in pairs(list) do
						inv_list[listname][index] = stack
					end
				end

				inv:set_lists(inv_list)
			end
		elseif args.type == "name" then
			args.value, success = lush.DatatypeValidator.cast_to(args.value, "string")

			if not success then
				error("invalid argument #3: invalid type")
			end

			for _, pos in args.pos:iterator() do
				local node = core.get_node(pos)
				core.set_node(pos, {name = args.value, param2 = node.param2, param1 = node.param1})
			end
		elseif args.type == "param1" then
			args.value, success = lush.DatatypeValidator.cast_to(args.value, "number")

			if not success then
				error("invalid argument #3: invalid type")
			end

			for _, pos in args.pos:iterator() do
				local node = core.get_node(pos)
				core.set_node(pos, {name = node.name, param2 = node.param2, param1 = args.value})
			end
		elseif args.type == "param2" then
			args.value, success = lush.DatatypeValidator.cast_to(args.value, "number")

			if not success then
				error("invalid argument #3: invalid type")
			end

			for _, pos in args.pos:iterator() do
				local node = core.get_node(pos)
				core.set_node(pos, {name = node.name, param2 = args.value, param1 = node.param1})
			end
		end
	end,
})
