-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

-- @classmod LanguageConstruct_InlineLua
--	Describes inline lua code chunks

local InlineLua =
{
	instance_of = "LanguageConstruct_InlineLua",
}

InlineLua.__index = InlineLua

function InlineLua:new(t)
	if t == nil then
		t = {}
	end

	-- {{{ type checking
	assert(type(t) == "table", "t should be of type table. instead of " .. type(t))
	if t.chunk and type(t.chunk) ~= "function" then error("t.chunk should be of type function. instead of " .. type(t.chunk)) end
	-- }}}

	self = setmetatable(
	{
		chunk = t.chunk or {}
	}, InlineLua)

	return self
end

local env_metatable = {}

function InlineLua:evaluate(ctx)
	-- {{{ type checking
	assert(ctx.instance_of == "ShellContext", "ctx should be an instance of ShellContext. instead of " .. ctx.instance_of)
	-- }}}

	local env_with_context = getfenv(1)
	setmetatable(env_with_context, env_metatable) -- remove the engine provided metatable, because it sends error warnings when setting globals
	env_with_context.ctx = ctx
	setfenv(self.chunk, env_with_context)
	return self.chunk(ctx)
end

function InlineLua:parse(parser_ctx)
	-- {{{ type checking
	assert(parser_ctx.instance_of == "ParserContext", "parser_ctx should be an instance of ParserContext. instead of " .. parser_ctx.instance_of)
	-- }}}

	-- state: *
	parser_ctx:advance()
	-- state: `*

	local offset_index = 1
	local char
	local is_escaped = false
	local code_text = ""
	local error_message
	while not parser_ctx:is_EOF() do
		char = parser_ctx:consume(offset_index)

		if char == "`" and not is_escaped then
			break
		elseif char == [[\]] then
			is_escaped = true
		elseif char == "n" and is_escaped then
			code_text = code_text .. "\n"
			is_escaped = false
		else
			if is_escaped then
				code_text = code_text .. [[\]]
				is_escaped = false
			end

			code_text = code_text .. char
		end

	end

	self.chunk, error_message = loadstring(code_text, "inline lua code")

	assert(self.chunk, error_message)

	-- state: `<code>`*
end

function InlineLua:dump(dump_ctx)
	local prefix = dump_ctx:color("(InlineLua) ", "ConstructSpecifier")
	dump_ctx:write_line(string.format("%s<lua bytecode>", prefix))
end

return InlineLua
