-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

require("tests/busted_definitions")
require("type_definitions")

describe("ParameterExpansion class: ",
function()
	local ShellContext = require("shell_context")
	local ParameterExpansion = require("language_constructs.parameter_expansion")
	local ParserContext = require("parser_context")
	local helpers = require("helpers")
	local shell_context_instance
	local parameter_expansion_instance
	local parser_context_instance

	before_each(
	function()
		shell_context_instance = ShellContext:new()
		parameter_expansion_instance = ParameterExpansion:new()
		parser_context_instance = ParserContext:new()

		shell_context_instance.env:set("parameter", "value")

	end)

	after_each(
	function()
		shell_context_instance.env:unset("undefined_parameter")
	end)

	it("constructs without errors",
	function()
		assert.has_no.errors(function() ParameterExpansion:new(shell_context_instance) end)
	end)

	describe("evaluates correctly ",
	function()

		it("without modifiers",
		function()
			parameter_expansion_instance.parameter_name = "parameter"

			assert.equals("value", parameter_expansion_instance:evaluate(shell_context_instance))
		end)

		it("without modifiers when given an undefined parameter",
		function()
			parameter_expansion_instance.parameter_name = "undefined_parameter"

			assert.equals(nil, parameter_expansion_instance:evaluate(shell_context_instance))
		end)
	end)

	describe("parse(): ",
	function()

		it("parses correctly when given the \"$var\" type of expansion",
		function()
			parameter_expansion_instance = ParameterExpansion:new(shell_context_instance)
			parser_context_instance.text = "$parameter"
			parameter_expansion_instance:parse(parser_context_instance)

			assert.has_properties_equaling(
			{
				parameter_name = "parameter",
			}, parameter_expansion_instance)


			assert.equals(11, parser_context_instance.character_index)

		end)

		it("parses correctly when given the \"${var}\" type of expansion",
		function()
			parameter_expansion_instance = ParameterExpansion:new(shell_context_instance)
			parser_context_instance.text = "${parameter}"
			parameter_expansion_instance:parse(parser_context_instance)

			assert.has_properties_equaling(
			{
				parameter_name = "parameter",
			}, parameter_expansion_instance)

			assert.equals(13, parser_context_instance.character_index)

		end)

	end)

	it("\"instance_of\" property is correctly set",
	function()
		assert.equals("LanguageConstruct_ParameterExpansion", parameter_expansion_instance.instance_of)
	end)
end)
