-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

require("tests/busted_definitions")

describe("LanguageConstruct_Expression",
function()
	local Expression = require("language_constructs.expression")
	local ShellContext = require("shell_context")
	local ParserContext = require("parser_context")
	local helpers = require("helpers")
	local shell_context_instance
	local parser_context_instance
	local expression_instance
	local terminators = helpers.create_trie({" ", "\t"})
	terminators = helpers.add_tries(terminators, helpers.control_operators_list)

	before_each(
	function()
		shell_context_instance = ShellContext:new()
		parser_context_instance = ParserContext:new()
		expression_instance = Expression:new()
	end)

	describe("parses correctly: ",
	function()
		it("when given a simple range",
		function()
			parser_context_instance.text = "<1:2>"

			expression_instance:parse(parser_context_instance, terminators)

			assert.equals(1,    expression_instance:evaluate(shell_context_instance).start)
			assert.equals(2,    expression_instance:evaluate(shell_context_instance).finish)
			assert.equals(true, expression_instance:evaluate(shell_context_instance).start_inclusive)
			assert.equals(true, expression_instance:evaluate(shell_context_instance).finish_inclusive)
		end)

		it("when given a range with an sub expression with > operator",
		function()
			parser_context_instance.text = "<1:((1 > 2) and 1 or 0)>"

			expression_instance:parse(parser_context_instance, terminators)

			assert.equals(1,    expression_instance:evaluate(shell_context_instance).start)
			assert.equals(0,    expression_instance:evaluate(shell_context_instance).finish)
			assert.equals(true, expression_instance:evaluate(shell_context_instance).start_inclusive)
			assert.equals(true, expression_instance:evaluate(shell_context_instance).finish_inclusive)
		end)

		it("when given a simple number",
		function()
			parser_context_instance.text = "10"

			expression_instance:parse(parser_context_instance, terminators)

			assert.equals(10, expression_instance:evaluate(shell_context_instance))
		end)

		it("when given a simple string",
		function()
			parser_context_instance.text = "hello"

			expression_instance:parse(parser_context_instance, terminators)

			assert.equals("hello", expression_instance:evaluate(shell_context_instance))
		end)

		it("when given a simple string that starts with a number",
		function()
			parser_context_instance.text = "12hello"

			expression_instance:parse(parser_context_instance, terminators)

			assert.equals("12hello", expression_instance:evaluate(shell_context_instance))
		end)

		it("when given a simple addition",
		function()
			parser_context_instance.text = "2 + 2"

			expression_instance:parse(parser_context_instance, terminators)

			assert.equals(4, expression_instance:evaluate(shell_context_instance))
		end)

		it("when given a simple addition (no spaces)",
		function()
			parser_context_instance.text = "2+2"

			expression_instance:parse(parser_context_instance, terminators)

			assert.equals(4, expression_instance:evaluate(shell_context_instance))
		end)

		it("when given a simple addition inside of paranthesis",
		function()
			parser_context_instance.text = "(2 + 2)"

			expression_instance:parse(parser_context_instance, terminators)

			assert.equals(4, expression_instance:evaluate(shell_context_instance))
		end)

		it("when given a tetriary like operation",
		function()
			parser_context_instance.text = "2 > 1 and 1 or 0"

			expression_instance:parse(parser_context_instance, terminators)

			assert.equals(1, expression_instance:evaluate(shell_context_instance))
		end)

		it("when given a tetriary like operation (2)",
		function()
			parser_context_instance.text = "2 < 1 and 1 or 0"

			expression_instance:parse(parser_context_instance, terminators)

			assert.equals(0, expression_instance:evaluate(shell_context_instance))
		end)

		it("when given a complex expression",
		function()
			parser_context_instance.text = "2 + 2 * 2"

			expression_instance:parse(parser_context_instance, terminators)

			assert.equals(6, expression_instance:evaluate(shell_context_instance))
		end)

		it("when given a complex expression (2)",
		function()
			parser_context_instance.text = "(2 + 2) * 2"

			expression_instance:parse(parser_context_instance, terminators)

			assert.equals(8, expression_instance:evaluate(shell_context_instance))
		end)

		it("when given a complex expression (3)",
		function()
			parser_context_instance.text = "(2 + 2) * 2^2"

			expression_instance:parse(parser_context_instance, terminators)

			assert.equals(16, expression_instance:evaluate(shell_context_instance))
		end)

		it("when given a complex expression (4)",
		function()
			parser_context_instance.text = "(2 + 2) * 2^2 + 1 + 2 + 3 * 6"

			expression_instance:parse(parser_context_instance, terminators)

			assert.equals(37, expression_instance:evaluate(shell_context_instance))
		end)

		it("when given a complex range expression",
		function()
			parser_context_instance.text = "<$i:$j>"

			expression_instance:parse(parser_context_instance, terminators)

			shell_context_instance.env:set("i", 5)
			shell_context_instance.env:set("j", 10)
			assert.equals(5,    expression_instance:evaluate(shell_context_instance).start)
			assert.equals(10,   expression_instance:evaluate(shell_context_instance).finish)
			assert.equals(true, expression_instance:evaluate(shell_context_instance).start_inclusive)
			assert.equals(true, expression_instance:evaluate(shell_context_instance).finish_inclusive)
		end)

		it("when given a complex vector expression",
		function()
			parser_context_instance.text = "(1 + 3, 2^6, (27 % 5) * 3)"

			expression_instance:parse(parser_context_instance, terminators)

			shell_context_instance.env:set("i", 5)
			shell_context_instance.env:set("j", 10)

			assert.equals(4,  expression_instance.members[1]:evaluate(shell_context_instance))
			assert.equals(64, expression_instance.members[2]:evaluate(shell_context_instance))
			assert.equals(6,  expression_instance.members[3]:evaluate(shell_context_instance))
		end)

		it("When given an [] operator",
		function()
			parser_context_instance.text = "$tab[1]"

			expression_instance:parse(parser_context_instance, terminators)

			shell_context_instance.env:set("tab", {1, 2, 3, 4, 5})

			assert.equals(1,  expression_instance:evaluate(shell_context_instance))
			assert.equals(8, parser_context_instance.character_index)
		end)

		it("When given nested [] operators",
		function()
			parser_context_instance.text = "$tab[$tab2[2]]"

			expression_instance:parse(parser_context_instance, terminators)

			shell_context_instance.env:set("tab", {a = 1, b = 2, c =  3})
			shell_context_instance.env:set("tab2", {"a", "b", "c"})

			assert.equals(2,  expression_instance:evaluate(shell_context_instance))
			assert.equals(15, parser_context_instance.character_index)
		end)

		it("When given [] operator with excessive spaces",
		function()
			parser_context_instance.text = "$tab[ 1 + 1 ]"

			expression_instance:parse(parser_context_instance, terminators)

			shell_context_instance.env:set("tab", {1, 2, 3, 4, 5})

			assert.equals(2,  expression_instance:evaluate(shell_context_instance))
			assert.equals(14, parser_context_instance.character_index)
		end)
	end)
end)
