-- Copyright (C) 2024 rstc
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

local helpers = require("helpers")
local operators = require("operators")
local Expression

local ParameterAsignment =
{
	instance_of = "LanguageConstruct_ParameterAsignment"
}

ParameterAsignment.__index = ParameterAsignment

function ParameterAsignment:new(t)
	-- {{{ type checking
	if t then
		assert(type(t) == "table", "t should be of type table. instead of " .. type(t))
	end
	-- }}}

	t = t or {}

	self = setmetatable(
	{
		-- @string lvalue
		--	the lvalue
		lvalue = t.lvalue,

		-- @param rvalue
		--	the rvalue that the lvalue will be set to
		rvalue = t.rvalue or ""
	}, ParameterAsignment)

	return self
end

function ParameterAsignment:evaluate(ctx)
	-- {{{ type checking
	assert(ctx.instance_of == "ShellContext", "ctx should be an instance of ShellContext. instead of " .. ctx.instance_of)
	-- }}}

	local evaluated_rvalue = self.rvalue:evaluate(ctx)
	if self.lvalue.instance_of == "LanguageConstruct_Expression" then
		-- Code copied from Expression.lua
		-- MAKE SURE IT DOESN"T GO OUT OF DATE

		-- elaborate way to recycle tables
		if not self.stack then
			self.stack = {}
		end
		self.stack[1] = nil

		local stack_head = 0
		-- Do all the operations except for the last
		for i = 1, #self.lvalue.expression - 1 do
			if self.lvalue.expression[i].instance_of then
				-- the element is a value
				stack_head = stack_head + 1
				self.stack[stack_head] = self.lvalue.expression[i]:evaluate(ctx)
			else
				-- the element is a value
				-- the element is an operator
				if self.lvalue.expression[i].type == "unary" then
					self.stack[stack_head] = self.lvalue.expression[i].action(self.stack[stack_head])
				else
					local result = self.lvalue.expression[i].action(self.stack[stack_head - 1], self.stack[stack_head])
					self.stack[stack_head - 1] = result
					self.stack[stack_head] = nil
					stack_head = stack_head - 1
				end
			end

		end

		local operator = self.lvalue.expression[#self.lvalue.expression]


		if operator == operators["["] then
			self.stack[1][self.stack[2]] = evaluated_rvalue
		elseif operator == operators["."] then
			self.stack[1][self.stack[2]] = evaluated_rvalue
		else
			error("Invalid parameter asignment")
		end

	elseif self.lvalue.instance_of == "LanguageConstruct_ParameterExpansion" then
		ctx.env:set(self.lvalue.parameter_name, evaluated_rvalue)
	end
end

local lvalue_terminators = helpers.create_trie({"="})

local white_chars_trie = helpers.create_trie({" ", "\t"})
local terminator_cache = {}

function ParameterAsignment:parse(parser_ctx, terminators)
	-- {{{ type checking
	assert(parser_ctx.instance_of == "ParserContext", "parser_ctx should be an instance of ParserContext. instead of " .. parser_ctx.instance_of)
	-- }}}

	Expression = Expression or require("language_constructs.expression")

	-- state: *

	if not self.lvalue then
		-- partial parsing
		self.lvalue = Expression:new()
		self.lvalue:parse(parser_ctx, lvalue_terminators)

		assert(self.lvalue.instance_of == "LanguageConstruct_ParameterExpansion" or self.lvalue.instance_of == "LanguageConstruct_Expression", "Invalid lvalue")
	end

	-- state: <parameter name>*
	parser_ctx:expect("=")
	-- state: <parameter name>=*

	local rvalue_terminators

	if terminator_cache[terminators] then
		rvalue_terminators = terminator_cache[terminators]
	else
		local tmp = helpers.add_tries(white_chars_trie, terminators)
		terminator_cache[terminators] = tmp
		rvalue_terminators = tmp
	end

	self.rvalue = Expression:new()
	self.rvalue:parse(parser_ctx, rvalue_terminators)

	-- state: <lvalue>=<rvalue>*
end

function ParameterAsignment:dump(dump_ctx)
	dump_ctx:write_text(dump_ctx:color("(ParameterAsignment)", "ConstructSpecifier"))
	dump_ctx:new_line()
	dump_ctx:write_text("[")
	dump_ctx:indent(1)
	dump_ctx:new_line()

	dump_ctx:write_text("lvalue: ")
	self.lvalue:dump(dump_ctx)
	dump_ctx:new_line()

	dump_ctx:write_text("rvalue: ")
	self.rvalue:dump(dump_ctx)

	dump_ctx:indent(-1)
	dump_ctx:new_line()
	dump_ctx:write_text("]")
end

return ParameterAsignment
