-- Copyright (C) 2025 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

local helpers = require("helpers")
local Shell = require("language_constructs.shell")
local ShellContext = require("shell_context")
local DatatypeValidator = require("datatype_validator")

local FunctionDefiniton =
{
	instance_of = "LanguageConstruct_FunctionDefinition",
}

FunctionDefiniton.__index = FunctionDefiniton

function FunctionDefiniton:new(t)
	t = t or {}
	self = setmetatable(
	{
		shell = t.shell,
		identifier = t.identifier,
		arg_rules = t.arg_rules
	}, FunctionDefiniton)

	return self
end

function FunctionDefiniton:evaluate(ctx)
	-- {{{ type checking
	assert(ctx.instance_of == "ShellContext", "ctx should be an instance of ShellContext. instead of " .. ctx.instance_of)
	-- }}}

	if self.identifier then
		ctx.env:set(self.identifier, self, false, false)
	else
		return self
	end
end

function FunctionDefiniton:__call(ctx, args)
	local subshell_ctx = ShellContext:new()
	subshell_ctx.privilege_cache = ctx.privilege_cache

	if self.arg_rules then
		local valildated_args = DatatypeValidator.validate(args, self.arg_rules, ctx)

		for k, v in pairs(valildated_args) do
			subshell_ctx.env:set(k, v, false, false)
		end
	end


	ctx:interrupt()

	self.shell:evaluate(ctx, {
		subshell_context = subshell_ctx,
		complete_callback = function(success, err_msg, ctx)
			-- assert(success, string.cormat("function %s: %s", self.identifier or "anonymous function", err_msg))
			if not success then
				ctx.exit_status = 1
				ctx.stderr = err_msg
			end

			ctx:stop_interruption()
		end
	})
end

local subshell_terminators = helpers.create_trie({"end"})

function FunctionDefiniton:parse(parser_ctx)
	-- {{{ type checking
	assert(parser_ctx.instance_of == "ParserContext", "parser_ctx should be an instance of ParserContext. instead of " .. parser_ctx.instance_of)
	-- }}}

	local arg_idx = 1

	local function parse_paramters()
		parser_ctx:expect("(")
		local char = parser_ctx:peek()

		if char == ")" then
			return parser_ctx:advance()
		end

		repeat
			local rule = {rule_type = "value", single = true}
			char = parser_ctx:peek()

			if not helpers.valid_identifier_chars_set[char] then
				error("invalid parameter name/type")
			end

			-- consume an identifier
			local start_idx = parser_ctx.character_index
			parser_ctx:skip_until(helpers.concat_sets(helpers.valid_identifier_chars_set, helpers.numeric_chars_set), true)
			local text = string.sub(parser_ctx.text, start_idx, parser_ctx.character_index - 1)

			char = parser_ctx:peek()

			if char == "[" then
				parser_ctx:expect("[") -- this should never fail, but whatever
				parser_ctx:expect("]")
				char = parser_ctx:peek()

				rule.single = false
			end

			if char == "," then
				if rule.single == false then
					-- if the rule is single, it means the parameter looks something like this: string[],
					-- this is a valid type, but its missing an identifier
					error("invalid parameter, name expected")
				end

				rule.name = text
			elseif char == ")" then
				if rule.single == false then
					-- if the rule is single, it means the parameter looks something like this: string[],
					-- this is a valid type, but its missing an identifier
					error("invalid parameter, name expected")
				end

				rule.name = text
			elseif helpers.white_chars_set[char] then
				parser_ctx:consume_whitespaces()

				char = parser_ctx:peek()

				assert(helpers.valid_identifier_chars_set[char], "invalid parameter name")

				rule = rule or {rule_type = "value", single = true}
				rule.type = text

				start_idx = parser_ctx.character_index
				parser_ctx:skip_until(helpers.concat_sets(helpers.valid_identifier_chars_set, helpers.numeric_chars_set), true)
				text = string.sub(parser_ctx.text, start_idx, parser_ctx.character_index - 1)

				rule.name = text
			else
				rule.name = text
			end

			char = parser_ctx:peek()

			if char == "?" then
				rule.optional = true
				parser_ctx:advance()
				char = parser_ctx:peek()
			end

			if char == "." then
				parser_ctx:expect(".")
				parser_ctx:expect(".")
				parser_ctx:expect(".")

				rule.rule_type = "block"
			end

			self.arg_rules = self.arg_rules or {}
			self.arg_rules[arg_idx] = rule

			parser_ctx:consume_whitespaces()
			char = parser_ctx:consume()
			parser_ctx:consume_whitespaces()
			assert(char == "," or char == ")", "parameters are expected to be terminated by commas")

			arg_idx = arg_idx + 1
		until char == ")"
	end

	parser_ctx:match_fast("function", true)
	parser_ctx:consume_whitespaces()

	local char = parser_ctx:peek()

	if char == "(" then
		parse_paramters()
	elseif helpers.valid_identifier_chars_set[char] then
		local start_idx = parser_ctx.character_index
		parser_ctx:skip_until(helpers.concat_sets(helpers.valid_identifier_chars_set, helpers.numeric_chars_set), true)

		self.identifier = string.sub(parser_ctx.text, start_idx, parser_ctx.character_index - 1)

		parser_ctx:consume_whitespaces()

		parse_paramters()
	else
		error("invalid function identifier")
	end

	parser_ctx:consume_whitespaces()

	self.shell = Shell:new()
	self.shell:parse(parser_ctx, {}, subshell_terminators)

	parser_ctx:advance(3)
	parser_ctx:consume_whitespaces()
end

function FunctionDefiniton:dump(dump_ctx)
	dump_ctx:write_text(dump_ctx:color("(FunctionDefiniton)", "ConstructSpecifier"))
	dump_ctx:new_line()
	dump_ctx:write_text("[")
	dump_ctx:indent(1)
	dump_ctx:new_line()

	if self.identifier then
		dump_ctx:write_text("identifier: " .. self.identifier)
		dump_ctx:new_line()
	end

	dump_ctx:write_text("Arg rules: ")
	dump_ctx:new_line()
	dump_ctx:write_text("[")
	dump_ctx:indent(1)

	if self.arg_rules then
		for k, v in pairs(self.arg_rules) do
			dump_ctx:new_line()
			dump_ctx:write_line(string.format("%s: %s", k, v))

		end
	end

	dump_ctx:indent(-1)
	dump_ctx:new_line()

	if self.shell then
		self.shell:dump(dump_ctx)
	end

	dump_ctx:indent(-1)
	dump_ctx:new_line()
	dump_ctx:write_text("]")
end

return FunctionDefiniton
