-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

local helpers = require("helpers")
local Expression = require("language_constructs.expression")
local Shell = require("language_constructs.shell")
local DatatypeValidator = require("datatype_validator")

local IfStatement =
{
	instance_of = "LanguageConstruct_IfStatement",
}

IfStatement.__index = IfStatement

function IfStatement:new(t)
	t = t or {}
	self = setmetatable(
	{
		expression = t.expression,
		success_subshell = t.success_subshell,
		elseif_expressions = t.elseif_expressions or {},
		elseif_subshells = t.elseif_subshells or {},
		fail_subshell = t.fail_subshell,
	}, IfStatement)

	return self
end

local function evaluate_as_bool(value)
	if type(value) == "table" and value.instance_of == "TypedList" then
		return #value.container ~= 0
	end

	local datatype = DatatypeValidator.get_type_of(value)

	if type(value) == "table" and value.instance_of == "TypedList" then
		return #value > 0
	elseif DatatypeValidator.registered_types[datatype]
		and DatatypeValidator.registered_types[datatype].to_bool then
		return DatatypeValidator.registered_types[datatype].to_bool(value)
	else
		return false
	end
end

function IfStatement:evaluate(ctx)
	-- {{{ type checking
	assert(ctx.instance_of == "ShellContext", "ctx should be an instance of ShellContext. instead of " .. ctx.instance_of)
	-- }}}


	local evaluated_expression = self.expression:evaluate(ctx)
	local success = evaluate_as_bool(evaluated_expression)

	ctx:interrupt()

	local function complete_callback()
		assert(ctx.exit_status == 0, "if statement: " .. ctx.stderr)

		ctx:stop_interruption()
	end

	if success then
		self.success_subshell:evaluate(ctx,
		{
			complete_callback = complete_callback
		})
	else
		for i = 1, #self.elseif_subshells do
			success = evaluate_as_bool(self.elseif_expressions[i]:evaluate(ctx))

			if success then
				self.elseif_subshells[i]:evaluate(ctx,
				{
					complete_callback = complete_callback
				})
				break
			end
		end

		if not success then
			if self.fail_subshell then
				self.fail_subshell:evaluate(ctx,
				{
					complete_callback = complete_callback
				})
			else
				ctx:stop_interruption()
			end
		end
	end

	assert(ctx.exit_status == 0, "if statement: " .. ctx.stderr)
end

local subshell_terminators = helpers.create_trie({"end", "else", "elseif"})
local last_subshell_terminator = helpers.create_trie({"end"})
local white_char_trie = helpers.create_trie({" ", "\t"})

function IfStatement:parse(parser_ctx)
	-- {{{ type checking
	assert(parser_ctx.instance_of == "ParserContext", "parser_ctx should be an instance of ParserContext. instead of " .. parser_ctx.instance_of)
	-- }}}
	local success

	-- state: *
	success = parser_ctx:match_fast("if", true)

	assert(success, "expected 'if'")

	parser_ctx:consume_whitespaces()

	-- state: if*

	self.expression = Expression:new()
	self.expression:parse(parser_ctx, white_char_trie)

	parser_ctx:consume_whitespaces()

	-- state: if <expression>*

	success = parser_ctx:match_fast("then", true)

	assert(success, "expected 'then'")

	-- state: if <expression> then*

	parser_ctx:consume_whitespaces()

	self.success_subshell = Shell:new()
	self.success_subshell:parse(parser_ctx, {}, subshell_terminators)

	-- state: if <expression> then
	-- <subshell>
	-- *

	local success, match = parser_ctx:match(subshell_terminators, true)

	assert(success, "expected any of 'end', 'else', 'elseif'")

	if match == "else" then
		-- state: if <expression> then
		-- <subshell>
		-- else
		-- *

		parser_ctx:consume_whitespaces()
		self.fail_subshell = Shell:new()
		self.fail_subshell:parse(parser_ctx, {}, last_subshell_terminator)

		-- state: if <expression> then
		-- <subshell>
		-- else
		-- <subshell>
		-- *

		parser_ctx:advance(3)
	elseif match == "elseif" then
		-- the state comments are becoming too verbose, im just going without them for this case
		local i = 1
		while match == "elseif" do
			parser_ctx:consume_whitespaces()

			self.elseif_expressions[i] = Expression:new()
			self.elseif_expressions[i]:parse(parser_ctx, white_char_trie)
			parser_ctx:consume_whitespaces()

			success = parser_ctx:match_fast("then", true)

			assert(success, "expected 'then'")

			parser_ctx:consume_whitespaces()
			self.elseif_subshells[i] = Shell:new()
			self.elseif_subshells[i]:parse(parser_ctx, {}, subshell_terminators)

			success, match = parser_ctx:match(subshell_terminators, true)

			assert(success, "expected any of 'end', 'else', 'elseif'")

			i = i + 1
		end

		if match == "else" then
			parser_ctx:consume_whitespaces()
			self.fail_subshell = Shell:new()
			self.fail_subshell:parse(parser_ctx, {}, last_subshell_terminator)
		end
	elseif match ~= "end" then
		error("unexplainable parser horrors")
	end

	-- state: if <expression> then
	-- <subshell>
	-- end*
end

function IfStatement:dump(dump_ctx)
	dump_ctx:write_text(dump_ctx:color("(IfStatement)", "ConstructSpecifier"))
	dump_ctx:new_line()
	dump_ctx:write_text("[")
	dump_ctx:indent(1)
	dump_ctx:new_line()

	dump_ctx:write_text("condition: ")
	self.expression:dump(dump_ctx)
	dump_ctx:new_line()
	dump_ctx:write_text("then: ")
	dump_ctx:new_line()
	self.success_subshell:dump(dump_ctx)
	dump_ctx:new_line()

	for i = 1, #self.elseif_subshells do
		dump_ctx:write_text("else condition: ")
		self.elseif_expressions[i]:dump(dump_ctx)
		dump_ctx:new_line()
		dump_ctx:write_text("then: ")
		dump_ctx:new_line()
		self.elseif_subshells[i]:dump(dump_ctx)
		dump_ctx:new_line()
	end

	if self.fail_subshell then
		dump_ctx:write_text("else: ")
		dump_ctx:new_line()
		self.fail_subshell:dump(dump_ctx)
		dump_ctx:new_line()
	end

	dump_ctx:indent(-1)
	dump_ctx:write_text("]")
end

return IfStatement
