-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

lush = {}

-- {{{ creating a compatibility require() 
local loaded_modules = {}
local current_directory = core.get_modpath("lush_core") .. "/src/"
local redirect_modules =
{
	["helpers"] = "minetest/helpers.lua",
	["command"] = "minetest/command.lua",
	["interpreter"] = "minetest/interpreter.lua",
	["dump_context"] = "minetest/dump_context.lua",
	["env"] = "minetest/env.lua",
	["shell_context"] = "minetest/shell_context.lua",
	["language_constructs.command_invocation"] = "minetest/language_constructs/command_invocation.lua",
	["language_constructs.vector_expression"] = "minetest/language_constructs/vector_expression.lua",
	["language_constructs.inline_lua"] = "minetest/language_constructs/inline_lua.lua",
	["type_definitions"] = "minetest/type_definitions.lua",
	["selector"] = "minetest/selector.lua",
	["language_constructs.while_loop"] = "minetest/language_constructs/while_loop.lua",
	["language_constructs.for_loop"] = "minetest/language_constructs/for_loop.lua",
}

local env_with_require = getfenv(1)

local require = function(module)
	local absolute_path = current_directory .. string.gsub(module, "%.", "/") .. ".lua"

	local function dofile_with_env(path)
		return setfenv(function()
			return dofile(path)
		end, env_with_require)()
	end

	-- if isnt loaded
	if not loaded_modules[module] then
		if redirect_modules[module] then
			local redirect_value = redirect_modules[module]
			redirect_modules[module] = nil -- to prevent recursion
			loaded_modules[module] = {dofile_with_env(current_directory .. redirect_value)}
		else
			loaded_modules[module] = {dofile_with_env(absolute_path)}
		end
	end

	return unpack(loaded_modules[module])
end
env_with_require.require = require
-- }}}

lush.debug = core and core.settings:get_bool("lush_debug") or false

lush.Storage = require("minetest.storage")
require("minetest.privs")
require("type_definitions")

lush.Env = require("env")
lush.Command, lush.RegisteredCommands = require("command")
lush.ShellContext = require("shell_context")
lush.ParserContext = require("parser_context")
lush.DatatypeValidator = require("datatype_validator")
lush.helpers = require("helpers")
lush.Selector = require("selector")
lush.Interpreter = require("interpreter")
lush.InventoryItemList = require("minetest.inventory_item_list")
lush.TypedList = require("typed_list")
lush.DumpContext = require("dump_context")

lush.global_env = lush.Env:new({parameters = lush.Storage.load("global_parameters", {})})
lush.spawnpoints = lush.Storage.load("spawnpoints", {})

local function save_data()
	lush.Storage.save("spawnpoints", lush.spawnpoints)
	-- lush.Storage.save("player_environments", lush.player_environments)
	
	local player_parameters = {}
	for player_name, player_env in pairs(lush.player_environments) do
		-- the player_env is wrapped in a env_stud, to get the actual env, get the env that the stud inherits from
		player_env = player_env.inherited_env 
		player_parameters[player_name] = {}

		for param, value in pairs(player_env.parameters) do
			if player_env.parameter_attributes[param]
				and bit.band(player_env.parameter_attributes[param], lush.helpers.ParamAttributes.Volatile) == 0 then
				player_parameters[player_name][param] = value
			end
		end
	end

	lush.Storage.save("player_parameters", player_parameters)

	local global_parameters = {}
	for param, value in pairs(lush.global_env.parameters) do
		if lush.global_env.parameter_attributes[param]
			and bit.band(lush.global_env.parameter_attributes[param], lush.helpers.ParamAttributes.Volatile) == 0 then

			global_parameters[param] = value
		end
	end

	lush.Storage.save("global_parameters", global_parameters)
end

local language_constructs =
{
	"bool",
	"command_invocation",
	"double_quote_string",
	"expression",
	"for_loop",
	"if_statement",
	"inline_lua",
	"number_literal",
	"parameter_asignment",
	"parameter_expansion",
	"process_substitution",
	"range_expression",
	"selector_expression",
	"shell",
	"single_quote_string",
	"string_expression",
	"table_expression",
	"unquoted_string",
	"vector_expression",
	"while_loop",
}

if lush.debug then
	for _, v in pairs(language_constructs) do
		local module = require("language_constructs." .. v)
		local old_parse = module.parse
		module.parse = function(self, parser_ctx, ...)
			self._start_index = parser_ctx.character_index
			old_parse(self, parser_ctx, ...)
			self._end_index = parser_ctx.character_index
		end

		local old_evaluate = module.evaluate

		module.evaluate = function(self, ctx, ...)
			ctx._last_execution = self
			return old_evaluate(self, ctx, ...)
		end
	end
end

core.register_on_shutdown(
function()
	save_data()
end)

core.register_on_mods_loaded(
function()
    core.register_on_respawnplayer(
	function(player)
        local name = player:get_player_name()
        if lush.spawnpoints[name] then
            player:set_pos(lush.spawnpoints[name])
            return true
        end
    end)
end)
