-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 


-- @fix this file requires the minetest namespace

local ShellContext = require("shell_context")
local Shell = require("language_constructs.shell")
local CommandInvocation = require("language_constructs.command_invocation")
local ParserContext = require("parser_context")
local DataTypeValidator = require("datatype_validator")
local helpers = require("helpers")

local Interpreter =
{
	instance_of = "Interpreter"
}

--- executes a shell
--	@string code the code to execute
--	@ShellContext initial_context the shell_context the shell will start with
--	@tab options a table with shell options: share_environment and complete_callback
--	@bool share_env whether the shell will share the environment of the passed ShellContext, this means that any parameter assignments will be persistent
--	@func complete_callback function called once execution ends
--	takes whether shell suceeded, error_message (when applicable) and shell context, parsing and evaluation time as arguments
function Interpreter.run_shell(code, initial_context, options)
	options.get_time = options.get_time or core.get_us_time -- the options.get_time() is used as a sort of dependency injection
	local t1 = options.get_time()
	if initial_context == nil then
		initial_context = {}
	end

	local shell_ctx = ShellContext:new(initial_context)
	-- shell_ctx.privilege_cache = minetest.get_player_privs(shell_ctx.env:get("name"))

	-- the shell would error if the first command returned a non string value, and the second command returned something that
	-- can't be turned into that. This ensures both can be converted into strings (all types can be converted to strings)
	shell_ctx.stdout.type = "string" 

	local parser_ctx = ParserContext:new({text = code})
	local shell = Shell:new()
	local success
	local message
	local parse_time
	local eval_time

	success, message = pcall(shell.parse, shell, parser_ctx, {}, {})

	if not success then
		options.complete_callback(false, message, shell_ctx)
		if lush.debug then
			helpers.print(parser_ctx)
			helpers.print(shell)
		end
		return
	end

	-- core.debug(helpers.dumpIR(shell))
	parse_time = options.get_time() - t1
	t1 = options.get_time()

	if options.complete_callback then
		local old_callback = options.complete_callback
		options.complete_callback = function(success, error_message, ctx)
			old_callback(ctx.exit_status == 0 and true or false, ctx.stderr, ctx, parse_time, options.get_time() - t1)
		end
	end

	success, message = pcall(shell.evaluate, shell, shell_ctx, options)

	if not success then
		options.complete_callback(false, message, shell_ctx)
	end
end

--- parses a shell, returns the parse tree that has the `evaluate(shell_ctx, options)` method.
--	@string code the code to parse
--	@treturn bool whether the parsing succeded
--	@treturn string error message, if any
--	@treturn tab the parse tree
--	@see Interpreter.run_shell
function Interpreter.parse_shell(code)

	local parser_ctx = ParserContext:new({text = code})
	local shell = Shell:new()
	local success
	local message

	success, message = pcall(shell.parse, shell, parser_ctx, {}, {})

	return success, message and string.format("parsing error: %s", message) or "parsing error: no error message", shell
end

--- similar to Interpreter.run_shell except it runs a single command instead of a shell
--	@string command
--	@ShellContext initial_context
--	@tab options
--	@treturn bool success
--	@treturn string error_message
--	@treturn number parse_time time it took to parse the command
--	@treturn number eval_time time it took to evaluate the command
function Interpreter.run_command(command, initial_context, options)
	local t1 = options.get_time()
	if initial_context == nil then
		initial_context = {}
	end

	local shell_ctx = ShellContext:new(initial_context)
	-- shell_ctx.privilege_cache = minetest.get_player_privs(shell_ctx.env:get("name"))

	local parser_ctx = ParserContext:new({text = command})
	local command_invocation = CommandInvocation:new()
	local success
	local message
	local parse_time
	local eval_time

	success, message = pcall(command_invocation.parse, command_invocation, parser_ctx, {})

	if not success then
		if lush.debug then
			helpers.print(parser_ctx)
			helpers.print(shell)
		end
		return false, message and string.format("parsing error: %s", message) or "parsing error: no error message", nil, nil
	end

	parse_time = options.get_time() - t1
	t1 = options.get_time()

	success, message = pcall(command_invocation.evaluate, command_invocation, shell_ctx)
	eval_time = options.get_time() - t1

	return success, message and string.format("evaluation error: %s", message),
		parse_time, eval_time
end

function Interpreter.run_file(path, initial_context, options)
	Interpreter.run_shell(io.open(path, "r"):read("*a"), initial_context, options)
end

-- local my_path = core.get_time() .. "/script.lush"
-- core.after(5, function()
-- 	Interpreter.run_file(my_path, {env = lush.player_environments["singleplayer"]}, {complete_callback = function(success, err_msg, ctx)
-- 		core.debug("out", err_msg, dump(ctx))
-- 	end})
-- end)

return Interpreter
