-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

---@diagnostic disable: param-type-mismatch

-- @classmod LanguageConstruct_RangeExpression
--	describes a vector

local helpers = require("helpers")
local Range = require("range")
local Expression

local RangeExpression =
{
	instance_of = "LanguageConstruct_RangeExpression",
}

RangeExpression.__index = RangeExpression

function RangeExpression:new(t)
	t = t or {}
	self = setmetatable(
	{
		start = t.members or -math.huge,
		finish = t.members or math.huge,
		start_inclusive = t.start_inclusive or true,
		finish_inclusive = t.finish_inclusive or true,
		static = t.static or false
	}, RangeExpression)

	return self
end

function RangeExpression:evaluate(ctx)
	-- {{{ type checking
	assert(ctx.instance_of == "ShellContext", "ctx should be an instance of ShellContext. instead of " .. ctx.instance_of)
	-- }}}
	local output = Range:new(
	{
		start_inclusive = self.start_inclusive,
		finish_inclusive = self.finish_inclusive
	})

	if self.start == -math.huge then
		output.start = -math.huge
	else
		output.start = self.start:evaluate(ctx)
	end

	if self.finish == math.huge then
		output.finish = math.huge
	else
		output.finish = self.finish:evaluate(ctx)
	end

	assert(type(output.start) == "number", "The start of a range needs to be a number")
	assert(type(output.finish) == "number", "The end of a range needs to be a number")

	return output
end

function RangeExpression:static_eval()
	local output = Range:new(
	{
		start_inclusive = self.start_inclusive,
		finish_inclusive = self.finish_inclusive
	})

	if self.start == -math.huge then
		output.start = -math.huge
	else
		output.start = self.start:static_eval()
	end

	if self.finish == math.huge then
		output.finish = math.huge
	else
		output.finish = self.finish:static_eval()
	end

	assert(type(output.start) == "number", "The start of a range needs to be a number")
	assert(type(output.finish) == "number", "The end of a range needs to be a number")

	return output
end

local first_subexpression_terminators = helpers.create_trie({":", " ", "\t"})
local second_subexpression_terminators = helpers.create_trie({")", ">", " ", "\t"})

-- the state comments dont include white spaces because it would be too verbose to be useful
function RangeExpression:parse(parser_ctx)
	-- {{{ type checking
	assert(parser_ctx.instance_of == "ParserContext", "parser_ctx should be an instance of ParserContext. instead of " .. parser_ctx.instance_of)
	-- }}}

	Expression = Expression or require("language_constructs.expression")
	self.start = -math.huge
	self.finish = math.huge

	local char = parser_ctx:consume()

	if char == "(" then
		self.start_inclusive = false
	elseif char == "<" then
		self.start_inclusive = true
	else
		error([[range expression expected to start with a "(" or "<", instead got ]] .. char)
	end

	-- state: <opening char>*

	parser_ctx:consume_whitespaces()
	char = parser_ctx:peek()

	if char ~= ":" then
		self.start = Expression:new()
		self.start:parse(parser_ctx, first_subexpression_terminators)
	end

	-- state: <opening char><numeric expression>*

	parser_ctx:consume_whitespaces()
	parser_ctx:expect(":")

	-- state: <opening char><numeric expression>:*

	parser_ctx:consume_whitespaces()
	char = parser_ctx:peek()

	if char ~= ")" and char ~= ">" then
		self.finish = Expression:new()
		self.finish:parse(parser_ctx, second_subexpression_terminators)
	end

	-- state: <opening char><numeric expression>:*<numeric expression>

	parser_ctx:consume_whitespaces()
	char = parser_ctx:consume()

	if char == ")" then
		self.finish_inclusive = false
	elseif char == ">" then
		self.finish_inclusive = true
	else
		error([[range expression expected to finish with a ")" or ">", instead got ]] .. char)
	end

	if self.start ~= -math.huge and self.start.static and self.finish ~= math.huge and self.finish.static then
		self.static = true
	end
end

function RangeExpression:dump(dump_ctx)
	dump_ctx:write_text(dump_ctx:color("(RangeExpression)", "ConstructSpecifier"))
	dump_ctx:new_line()
	dump_ctx:write_text("[")
	dump_ctx:indent(1)
	dump_ctx:new_line()

	dump_ctx:write_text(string.format("Start is inclusive: %s", tostring(self.start_inclusive)))
	dump_ctx:new_line()
	dump_ctx:write_text(string.format("Finish is inclusive: %s", tostring(self.finish_inclusive)))
	dump_ctx:new_line()
	dump_ctx:write_text("Start: ")
	if self.start == -math.huge then
		dump_ctx:write_text("<negative infinity>")
	else
		self.start:dump(dump_ctx)
	end
	dump_ctx:new_line(1)

	dump_ctx:write_text("Finish: ")
	if self.finish == math.huge then
		dump_ctx:write_line("<infinity>")
	else
		self.finish:dump(dump_ctx)
	end

	dump_ctx:indent(-1)
	dump_ctx:new_line()
	dump_ctx:write_text("]")
end

return RangeExpression
