-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

--- @classmod LanguageConstruct_SelectorExpression
--	describes a vector

local selector = require("selector")
local TableExpression
local TypedList


local SelectorExpression =
{
	instance_of = "LanguageConstruct_SelectorExpression",
}

SelectorExpression.__index = SelectorExpression

function SelectorExpression:new(t)
	t = t or {}
	self = setmetatable(
	{
		preset = t.preset or {},
		explicit_parameters = t.explicit_parameters or {}
	}, SelectorExpression)

	return self
end

function SelectorExpression:evaluate(ctx)
	-- {{{ type checking
	assert(ctx.instance_of == "ShellContext", "ctx should be an instance of ShellContext. instead of " .. ctx.instance_of)
	-- }}}

	TypedList = TypedList or require("typed_list")

	local evaluated_explicit_params
	if self.explicit_parameters.instance_of == "LanguageConstruct_TableExpression" then
		evaluated_explicit_params = self.explicit_parameters:evaluate(ctx)
	end

	local expanded_list = selector.expand(self.preset, evaluated_explicit_params, ctx)
	local output = TypedList:new()

	output:concat(expanded_list)

	return output
end

-- @fixme add infinite ranges support
-- the state comments dont include white spaces because it would be too verbose to be useful
function SelectorExpression:parse(parser_ctx)
	-- {{{ type checking
	assert(parser_ctx.instance_of == "ParserContext", "parser_ctx should be an instance of ParserContext. instead of " .. parser_ctx.instance_of)
	-- }}}

	TableExpression = TableExpression or require("language_constructs.table_expression")

	parser_ctx:expect("@")

	self.preset = parser_ctx:consume()

	assert(self.preset, "invalid selector")

	local char = parser_ctx:peek()

	if char == "{" then
		self.explicit_parameters = TableExpression:new()
		self.explicit_parameters:parse(parser_ctx)
	end

end

function SelectorExpression:dump(dump_ctx)
	dump_ctx:write_line(dump_ctx:color("(SelectorExpression)", "ConstructSpecifier"))
	dump_ctx:write_text("[")
	dump_ctx:indent(1)

	dump_ctx:write_line(string.format("Selector preset: %s", self.preset))
	if #self.explicit_parameters ~= 0 then
		dump_ctx:write_line("Parameters: ", true)
		self.explicit_parameters:dump(dump_ctx)
	end

	dump_ctx:indent(-1)
	dump_ctx:new_line()
	dump_ctx:write_text("]")

end

return SelectorExpression
