-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

--- @classmod LanguageConstruct_TableExpression
--	describes a vector

local helpers = require("helpers")
local Expression
local StringExpression

local TableExpression =
{
	instance_of = "LanguageConstruct_TableExpression",
}

TableExpression.__index = TableExpression

function TableExpression:new(t)
	t = t or {}
	self = setmetatable(
	{
		-- key_container = t.key_container or {},
		-- value_container = t.value_container or {}
		container = t.container or {}
	}, TableExpression)

	return self
end

function TableExpression:evaluate(ctx)
	-- {{{ type checking
	assert(ctx.instance_of == "ShellContext", "ctx should be an instance of ShellContext. instead of " .. ctx.instance_of)
	-- }}}

	local output = {}

	for key, value in pairs(self.container) do
		if type(key) ~= "number" then
			key = key:evaluate(ctx)

			assert(key ~= "instance_of", "'instance_of' is a restricted table key")
		end

		output[key] = value:evaluate(ctx)
	end

	return output
end

local key_terminators = helpers.create_trie({"\n", "\t", " ", ",", "=", "}"})
local value_terminators = helpers.create_trie({"\n", "\t", " ", ",", "}"})

-- @fixme add infinite ranges support
-- the state comments dont include white spaces because it would be too verbose to be useful
function TableExpression:parse(parser_ctx)
	-- {{{ type checking
	assert(parser_ctx.instance_of == "ParserContext", "parser_ctx should be an instance of ParserContext. instead of " .. parser_ctx.instance_of)
	-- }}}

	Expression = Expression or require("language_constructs.expression")
	StringExpression = StringExpression or require("language_constructs.string_expression")

	local char = parser_ctx:consume()
	local key
	local value

	assert(char == "{", [[table is expected to start with "{", instead of ]] .. char)

	while true do
		parser_ctx:consume_whitespaces(true)

		key = Expression:new()
		key:parse(parser_ctx, key_terminators)

		-- state: <name>*

		parser_ctx:consume_whitespaces(true)

		char = parser_ctx:peek()

		if char == "=" then
			if key.instance_of ~= "LanguageConstruct_DoubleQuoteString"
				and key.instance_of ~= "LanguageConstruct_SingleQuoteString"
				and key.instance_of ~= "LanguageConstruct_UnquotedString" then
				error("table key has to be a string")
			end

			parser_ctx:advance()
			parser_ctx:consume_whitespaces(true)

			-- state: <name>=*

			value = Expression:new()
			value:parse(parser_ctx, value_terminators)

			-- state: <name>=<value>*

			parser_ctx:consume_whitespaces(true)

			self.container[key] = value
		elseif char == "," or char == "}" then
			table.insert(self.container, key)
		else
			error("unexpected character: " .. char)
		end

		char = parser_ctx:consume()

		if char == "}" then
			break
		elseif char ~= "," then
			error("unexpected character " .. char)
		end
	end
end

function TableExpression:dump(dump_ctx)
	dump_ctx:write_text(dump_ctx:color("(TableExpression)", "ConstructSpecifier"))
	dump_ctx:new_line()
	dump_ctx:write_text("[")
	dump_ctx:indent(1)
	dump_ctx:new_line()

	for k, v in pairs(self.container) do
		if type(k) ~= "number" then
			k:dump(dump_ctx)
			dump_ctx:write_text("=")
		end
		v:dump(dump_ctx)
		dump_ctx:new_line()
	end

	dump_ctx:indent(-1)
	dump_ctx:new_line()
	dump_ctx:write_text("]")
end

return TableExpression
