-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

describe("helper functions: ",
function()
	local helpers = require("helpers")

	describe("is_option(): ",
	function()
		local is_valid_option, option_name

		it("recognizes single hypen options",
		function()
			is_valid_option, option_name = helpers.is_option("!a")

			assert.is.equal("a", option_name)
			assert.is_true(is_valid_option)
		end)

		it("recognizes double hypen options",
		function()
			is_valid_option, option_name = helpers.is_option("!option")

			assert.is.equal("option", option_name)
			assert.is_true(is_valid_option)
		end)

		it("dosent count option names that have special characters",
		function()
			is_valid_option, option_name = helpers.is_option("!$option")

			assert.is.equal(nil, option_name)
			assert.is_false(is_valid_option)
		end)

		it("dosent count options that start with a numbers",
		function()
			is_valid_option, option_name = helpers.is_option("!12option")

			assert.is.equal(nil, option_name)
			assert.is_false(is_valid_option)
		end)

		it("counts if contains numbers in the middle",
		function()
			is_valid_option, option_name = helpers.is_option("!option2")

			assert.is.equal("option2", option_name)
			assert.is_true(is_valid_option)
		end)
	end)

	describe("is_vector(): ",
	function()
		local vec

		it("dosent count n > 3 dimensional vectors",
		function()
			vec = {1, 2, 3, 4}

			assert.is_false(helpers.is_vector(vec))
		end)

		it("dosent count n < 3 dimensional vectors",
		function()
			vec = {1, 2}

			assert.is_false(helpers.is_vector(vec))
		end)

		it("dosent count if any member isnt a number",
		function()
			vec = {"1", "2", "3"}

			assert.is_false(helpers.is_vector(vec))
		end)

		it("returns true if given a valid vector",
		function()
			vec = {1, 2, 3}

			assert.is_true(helpers.is_vector(vec))
		end)

		it("returns false if givan a string",
		function()
			assert.is_false(helpers.is_vector("string"))
		end)
	end)

	describe("is_valid_identifier_char(): ",
	function()

		it("returns true when given a valid char ",
		function()
			assert.is(true, helpers.is_valid_identifier_char("a"))
		end)

		it("returns false when given a invalid character ",
		function()
			assert.is(true, helpers.is_valid_identifier_char("-"))
		end)

		it("returns false when given a numeric character as the leading character",
		function()
			assert.is(true, helpers.is_valid_identifier_char("1", true))
		end)

		it("returns false when given a valid character as the leading character",
		function()
			assert.is(true, helpers.is_valid_identifier_char("a", true))
		end)

		it("errors when given a string instead of a single character",
		function()
			assert.errors(function() helpers.is_valid_identifier_char("string") end)
		end)
	end)

	describe("is_single_character(): ",
	function()

		it("returns true when given a single character",
		function()
			assert.is(true, helpers.is_single_character("a"))
		end)

		it("returns true when given an empty string",
		function()
			assert.is(true, helpers.is_single_character(""))
		end)

		it("returns false when given a string with more than one character",
		function()
			assert.is(false, helpers.is_single_character("string"))
		end)

		it("errors when given nonstring argument",
		function()
			assert.errors(function() helpers.is_single_character(123) end)
		end)
	end)

	describe("is_valid_identifier_name(): ",
	function()

		it("returns true when given a valid name",
		function()
			assert.is(false, helpers.is_valid_identifier_name("name_123"))
		end)

		it("returns false when given an invalid name",
		function()
			assert.is(false, helpers.is_valid_identifier_name("name$$123"))
		end)

		it("returns false when given an invalid char as the leading char",
		function()
			assert.is(false, helpers.is_valid_identifier_name("!name"))
		end)

		it("returns false when given a leading number",
		function()
			assert.is(false, helpers.is_valid_identifier_name("123name"))
		end)

		it("errors when given a wrong argument",
		function()
			assert.errors(function() helpers.is_valid_identifier_char(123) end)
		end)
	end)


	describe("helpers.helpers.string_split(): ",
	function()

		it("correctly splits a string into a list of words",
		function()
			assert.same({"word", "list"}, helpers.string_split("word list", " "))
		end)

		it("returns a table with an empty string when given a empty string",
		function()
			assert.same({""}, helpers.string_split("", " "))
		end)

		it("returns a table with one element when given a string that cant be split",
		function()
			assert.same({"word list"}, helpers.string_split("word list", "|"))
		end)

		it("dosent add empty elements to the output table when given consecutive seperators and allow_empty_elements is false",
		function()
			assert.same({"word", "list"}, helpers.string_split("word   list", " ", false))
		end)

		it("dosent add empty elements to the output table when given trailing seperator and allow_empty_elements is false",
		function()
			assert.same({"word", "list"}, helpers.string_split("word list ", " ", false))
		end)

		it("dosent add empty elements to the output table when given leading seperator and allow_empty_elements is false",
		function()
			assert.same({"word", "list"}, helpers.string_split(" word list", " ", false))
		end)

		it("adds empty elements to the output table when given consecutive seperators and allow_empty_elements is true",
		function()
			assert.same({"word", "", "", "list"}, helpers.string_split("word   list", " ", true))
		end)

		it("adds empty elements to the output table when given trailing seperator and allow_empty_elements is true",
		function()
			assert.same({"word", "list", ""}, helpers.string_split("word list ", " ", true))
		end)

		it("adds empty elements to the output table when given leading seperator and allow_empty_elements is true",
		function()
			assert.same({"", "word", "list"}, helpers.string_split(" word list", " ", true))
		end)

		it("errors when given wrong arguments",
		function()
			assert.errors(function() helpers.string_split(123, {"table"}) end)
		end)
	end)
end)
