-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

require("tests/busted_definitions")

describe("LanguageConstruct_DoubleString class:",
function()
	local DoubleQuoteString = require("language_constructs.double_quote_string")
	local ParameterExpansion = require("language_constructs.parameter_expansion")
	local ShellContext = require("shell_context")
	local ParserContext = require("parser_context")
	local shell_context_instance
	local parser_context_instance
	local double_quote_string_instance

	before_each(
	function()
		shell_context_instance = ShellContext:new()
		shell_context_instance.env:set("parameter", "value")
		parser_context_instance = ParserContext:new()
		double_quote_string_instance = DoubleQuoteString:new()
	end)

	describe("evaluates correctly: ",
	function()

		it("when given a simple string",
		function()
			parser_context_instance.text = [[""]]

			double_quote_string_instance:parse(parser_context_instance)

			assert.equals("", double_quote_string_instance:evaluate(shell_context_instance))
		end)

		it("when given a simple string",
		function()
			double_quote_string_instance.elements = {"string"}

			assert.equals("string", double_quote_string_instance:evaluate(shell_context_instance))
		end)

		it("when given a string with an expansion",
		function()
			double_quote_string_instance.elements = {"string", ParameterExpansion:new({parameter_name = "parameter"})}

			shell_context_instance.env:set("parameter", "value")

			assert.equals("stringvalue", double_quote_string_instance:evaluate(shell_context_instance))
		end)

		it("when given a string with an expansion at the end",
		function()
			parser_context_instance.text = [["text${parameter}"]]

			double_quote_string_instance:parse(parser_context_instance)

			assert.equals("textvalue", double_quote_string_instance:evaluate(shell_context_instance))
		end)

		it("when given a string with an expansion in the middle of the string",
		function()
			double_quote_string_instance.elements = {"string", ParameterExpansion:new({parameter_name = "parameter"}), "string"}

			shell_context_instance.env:set("parameter", "value")

			assert.equals("stringvaluestring", double_quote_string_instance:evaluate(shell_context_instance))
		end)
	end)

	describe("parses correctly: ",
	function()

		it("when given a simple string",
		function()
			parser_context_instance.text = [["string"]]

			double_quote_string_instance:parse(parser_context_instance)

			assert.same({"string"}, double_quote_string_instance.elements)

			assert.equals(9, parser_context_instance.character_index)
		end)

		it("when given a string with an expansion",
		function()
			parser_context_instance.text = [["string${parameter}string"]]


			double_quote_string_instance:parse(parser_context_instance)

			assert.equals("string", double_quote_string_instance.elements[1])

			assert.equals("parameter", double_quote_string_instance.elements[2].parameter_name)

			assert.equals("string", double_quote_string_instance.elements[3])

			assert.equals(3, #double_quote_string_instance.elements)
		end)

		it("respects escape sequences on $ character",
		function()
			parser_context_instance.text = [["string\${parameter}"]]

			double_quote_string_instance:parse(parser_context_instance)

			assert.same({"string${parameter}"}, double_quote_string_instance.elements)

			assert.equals(22, parser_context_instance.character_index)

			assert.equals(1, #double_quote_string_instance.elements)
		end)

		it("respects escape sequences on \" character",
		function()
			parser_context_instance.text = [["\"string\""]]

			double_quote_string_instance:parse(parser_context_instance)

			assert.same({"\"string\""}, double_quote_string_instance.elements)

			assert.equals(13, parser_context_instance.character_index)

			assert.equals(1, #double_quote_string_instance.elements)
		end)

		it("respects escape sequences on basckslashes",
		function()
			parser_context_instance.text = [["string\\"]]

			double_quote_string_instance:parse(parser_context_instance)

			assert.equals([[string\]], double_quote_string_instance.elements[1])

			assert.equals(11, parser_context_instance.character_index)

			assert.equals(1, #double_quote_string_instance.elements)
		end)

	end)

	describe("members are correctly set by constructor: ",
	function()
		it("elements",
		function()
			double_quote_string_instance = DoubleQuoteString:new({elements = {"string"}})

			assert.same({"string"}, double_quote_string_instance.elements)
		end)

		it("instance_of",
		function()
			double_quote_string_instance = DoubleQuoteString:new()

			assert.equals("LanguageConstruct_DoubleQuoteString", double_quote_string_instance.instance_of)
		end)
	end)

end)
