-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

-- this file contains simple commands that dont depend on core namespace

core.register_privilege("export",
{
	description = "whether the player can modify lush's global environment",
	give_to_singleplayer = true
})

lush.Command:new(
{
	name = "echo",
	description = "prints arguments to STDOUT",
	params = "<arguments>",
	callback = function(ctx, args, options)
		for _, v in pairs(args) do
			if type(v) == "table" and v.instance_of == "TypedList" then
				ctx.stdout:concat(v)
			else
				ctx.stdout:push(v)
			end
		end
	end,
})

lush.Command:new(
{
	name = "modpath",
	description = "prints the absolute path of a given mod",
	privs = {server = true},
	arguments =
	{
		{
			name = "mod name",
			type = "string",
			single = true
		}
	},
	callback = function(ctx, args, options)
		ctx.stdout:push(core.get_modpath(args["mod name"]))
	end
})

lush.Command:new(
{
	name = "sleep",
	description = "interrupts the shell for N seconds",
	arguments =
	{
		{
			name = "time",
			type = "time",
			single = true
		}
	},
	callback = function(ctx, args, options)
		ctx:interrupt()
		core.after(args.time, ctx.stop_interruption, ctx)
	end,
})

lush.Command:new(
{
	name = "type",
	description = "prints the type of the argument specified",
	arguments =
	{
		{
			name = "value",
			single = true
		}
	},
	callback = function(ctx, args, options)
		ctx.stdout:push(lush.DatatypeValidator.get_type_of(args.value))
	end,
})

lush.Command:new(
{
	name = "cast",
	description = "casts a value to the specified type",
	arguments =
	{
		{
			name = "value",
		},
		{
			rule_type = "terminal",
			value = "to",
			optional = true
		},
		{
			name = "type",
			type = "string",
			single = true
		}
	},
	callback = function(ctx, args, options)
		local success = args.value:cast_to(args.type, ctx)

		if not success then
			error(string.format("cant convert from '%s' to '%s'", args.value.type, args.type))
		end

		ctx.stdout = args.value
	end,
})

lush.Command:new(
{
	name = "append",
	description = "appends a string to every line in STDIN",
	arguments =
	{
		{
			name = "postfix",
			type = "string",
			single = true
		}
	},
	stdin_type = "string",
	callback = function(ctx, args, options)
		for _, v in pairs(ctx.stdin.container) do
			ctx.stdout:push(v .. args.postfix)
		end
	end,
})

lush.Command:new(
{
	name = "lines",
	description = "takes STDIN and prints only lines in the range",
	arguments =
	{
		{
			name = "range",
			type = "range",
			single = true
		}
	},
	callback = function(ctx, args, options)
		for line_num, line in ctx.stdin:iterator() do
			if args.range:contains(line_num) then
				ctx.stdout:push(line)
			end
		end
	end,
})

lush.Command:new(
{
	name = "wc",
	description = "counts the amount of lines in STDIN",
	arguments = {},
	callback = function(ctx, args, options)
		ctx.stdout:push(#ctx.stdin.container)
	end,
})

lush.Command:new(
{
	name = "prepend",
	description = "prepends all lines from STDIN with the given string",
	arguments =
	{
		{
			name = "prefix",
			type = "string",
			single = true
		}
	},
	stdin_type = "string",
	callback = function(ctx, args, options)
		for _, v in pairs(ctx.stdin.container) do
			ctx.stdout:push(args.prefix .. v)
		end
	end,
})

lush.Command:new(
{
	name = "set",
	description = "sets the given parameter to the value of STDIN",
	arguments =
	{
		{
			name = "param",
			type = "string",
			single = true
		}
	},
	callback = function(ctx, args, options)
		ctx.env:set(args.param, ctx.stdin:get(1))
	end,
})

lush.Command:new(
{
	name = "unset",
	description = "unsets the given parameter",
	arguments =
	{
		{
			name = "param",
			type = "string",
			single = true
		}
	},
	callback = function(ctx, args, options)
		ctx.env:unset(args.param)
	end,
})

lush.Command:new(
{
	name = "split",
	description = "splits STDIN into lines",
	arguments =
	{
		{
			name = "seperator",
			type = "string",
			single = true
		}
	},
	stdin_type = "string",
	callback = function(ctx, args, options)
		for _, v in pairs(ctx.stdin.container) do
			ctx.stdout:concat(lush.helpers.string_split(v, args.seperator))
		end
	end,
})

lush.Command:new(
{
	name = "match",
	description = "prints only the lines from STDIN that match the pattern",
	arguments =
	{
		{
			name = "pattern",
			type = "string",
			single = true
		}
	},
	stdin_type = "string",
	callback = function(ctx, args, options)
		for _, v in pairs(ctx.stdin.container) do
			if string.find(v, args.pattern) then
				ctx.stdout:push(v)
			end

		end
	end,
})

lush.Command:new(
{
	name = "exit",
	description = "exits with the specified exit code",
	options = {errmsg = true},
	arguments =
	{
		{
			name = "exit_status",
			type = "number",
			single = true
		}
	},
	callback = function(ctx, args, options)
		if args.exit_status ~= 0 then
			error(options.errmsg)
		end
	end,
})

lush.Command:new(
{
	name = "veccopy",
	description = "creates a copy of a vector",
	arguments =
	{
		{
			name = "vector",
			type = "vector",
			single = true
		}
	},
	callback = function(ctx, args, options)
		ctx.stdout:push(args.vector:copy())
	end
})

lush.Command:new(
{
	name = "vecdesc",
	description = "verbosely prints information about a vector",
	arguments =
	{
		{
			name = "vector",
			type = "vector",
			single = true
		}
	},
	callback = function(ctx, args, options)
		ctx.stdout:push("x: " .. tostring(args.vector.x))
		ctx.stdout:push("y: " .. tostring(args.vector.y))
		ctx.stdout:push("z: " .. tostring(args.vector.z))
	end,
})

lush.Command:new(
{
	name = "export",
	description = "exports the parameter to the global environment, where every player can access it. if no value is provided, it unsets the global variable instead",
	options =
	{
		["read-only"] = false,
		["volatile"] = false,
	},
	privs = {export = true},
	arguments =
	{
		{
			name = "param",
			type = "string",
			single = true
		},
		{
			name = "value",
			single = true
		},
	},
	callback = function(ctx, args, options)
			lush.global_env:set(args.param, args.value, options["read-only"], options["volatile"])
	end,
})

lush.Command:new(
{
	name = "env",
	description = "prints the entire environment to STDOUT",
	callback = function(ctx, args, options)
		if #args ~= 0 then
			error("invalid arguments: too many arguments")
		end

		local function print_env(ctx, env)
			if env.inherited_env and env.inherited_env.instance_of == "Env" then
				print_env(ctx, env.inherited_env)
			end

			for k, v in pairs(env.parameters) do
				ctx.stdout:push(string.format("%s = %s", k, lush.DatatypeValidator.cast_to(v, "string", ctx)))
			end
		end

		print_env(ctx, ctx.env)
	end,
})

lush.Command:new(
{
	name = "void",
	description = "voids STDIN passed into the command",
	arguments = {},
	callback = function(ctx, args, options)
	end,
})

lush.Command:new(
{
	name = "tableunpack",
	description = "Takes a table value and converts it into TypedList of elements",
	arguments =
	{
		{
			name = "table",
			type = "table",
			single = true
		}
	},
	callback = function(ctx, args, options)
		ctx.stdout:concat(args.table)
	end,
})

lush.Command:new(
{
	name = "randomelem",
	description = "Prints a random element from a table (only arrays!). When using `amount` option, the elements can repeat",
	options = {amount = true},
	arguments =
	{
		{
			name = "tab",
			single = false
		}
	},
	callback = function(ctx, args, options)
		-- special handling for notmal tables
		local tab
		if args.tab.type == "table" then
			tab = args.tab:get(1)
		else
			tab = args.tab.container
		end

		for _ = 1, options.amount do
			ctx.stdout:push(tab[math.random(1, #tab)])
		end
	end,
})

lush.Command:new(
{
	name = "seq",
	description = "prints a sequence of numbers",
	arguments =
	{
		{
			name = "start",
			type = "number",
			single = true,
		},
		{
			name = "finish",
			type = "number",
			single = true,
		},
		{
			name = "step",
			type = "number",
			single = true,
		},
	},
	callback = function(ctx, args, options)
		for i = args.start, args.finish, args.step or 1 do
			ctx.stdout:push(i)
		end
	end
})
