-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

--- @classmod LanguageConstruct_NumberLiteral
--	describes a numeric expression

local helpers = require("helpers")

local NumberLiteral =
{
	instance_of = "LanguageConstruct_NumberLiteral",
	static = true,
}

NumberLiteral.__index = NumberLiteral

function NumberLiteral:new(t)
	t = t or {}

	-- {{{ type checking
	assert(type(t) == "table", "t should be of type table. instead of " .. type(t))
	if t.number and type(t.number) ~= "number" then error("t.number should be of type number. instead of " .. type(t.number)) end
	-- }}}
	self = setmetatable(
	{
		value = t.value
	}, NumberLiteral)

	return self
end

function NumberLiteral:evaluate(ctx)
	-- {{{ type checking
	assert(ctx.instance_of == "ShellContext", "ctx should be an instance of ShellContext. instead of " .. ctx.instance_of)
	-- }}}

	return self.value
end

function NumberLiteral:static_eval()
	return self.value
end

function NumberLiteral:parse(parser_ctx)
	NumberLiteral = NumberLiteral or require("language_constructs.number_literal")

	local start_index = parser_ctx.character_index
	local char
	local fraction_point = false -- used to make sure only one fraction dot is used

	if parser_ctx:peek() == "-" then
		parser_ctx:advance()
	end

	while true do
		char = parser_ctx:peek()

		if char == "." then
			assert(not fraction_point, "too many dots!")
			fraction_point = true
		elseif(not helpers.numeric_chars_set[char]) then
			break
		end

		parser_ctx:advance()
	end

	-- using lua's builtin parsing functionality
	-- why work hard when you can work smart?
	self.value = tonumber(string.sub(parser_ctx.text, start_index, parser_ctx.character_index - 1))

	assert(self.value, "invalid number format")
end

function NumberLiteral:dump(dump_ctx)
	local prefix = dump_ctx:color("(NumberLiteral) ", "ConstructSpecifier")
	dump_ctx:write_text(string.format("%s%s", prefix, dump_ctx:color(self.value, "Number")))
end

return NumberLiteral
