-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

--- @classmod LanguageConstruct_ProcessSubstitution
--	describes a subshell similar to bash's process substitution

local helpers = require("helpers")
local Shell
local ShellContext

local ProcessSubstitution =
{
	instance_of = "LanguageConstruct_ProcessSubstitution",
}

ProcessSubstitution.__index = ProcessSubstitution

function ProcessSubstitution:new(t)
	t = t or {}
	self = setmetatable(
	{
		subshell = t.subshell
	}, ProcessSubstitution)

	return self
end

function ProcessSubstitution:evaluate(ctx)
	-- {{{ type checking
	assert(ctx.instance_of == "ShellContext", "ctx should be an instance of ShellContext. instead of " .. ctx.instance_of)
	-- }}}

	ShellContext = ShellContext or require("shell_context")
	local subshell_ctx = ShellContext:new()
	subshell_ctx.env:inherit(ctx.env)
	subshell_ctx.privilege_cache = ctx.privilege_cache

	self.subshell:evaluate(subshell_ctx, {share_environment = true})

	if #subshell_ctx.stdout.container == 1 then
		return subshell_ctx.stdout.container[1]
	else
		return subshell_ctx.stdout
	end
end

local subshell_terminator = helpers.create_trie({")"})

function ProcessSubstitution:parse(parser_ctx)
	-- {{{ type checking
	assert(parser_ctx.instance_of == "ParserContext", "parser_ctx should be an instance of ParserContext. instead of " .. parser_ctx.instance_of)
	-- }}}

	local Shell = Shell or require("language_constructs.shell")

	parser_ctx:expect("$")
	parser_ctx:expect("(")

	self.subshell = Shell:new()
	self.subshell:parse(parser_ctx, {}, subshell_terminator)
end

function ProcessSubstitution:dump(dump_ctx)
	dump_ctx:write_text(dump_ctx:color("(ProcessSubstitution)", "ConstructSpecifier"))
	dump_ctx:new_line()
	self.subshell:dump(dump_ctx)
end

return ProcessSubstitution
