local commands = {}
local S = conquer.S

function conquer.register_command(name, def)
	def.name = name
	commands[name] = def
end

minetest.register_chatcommand("conquer", {
	func = function(name, param)
		for cmd_name, def in pairs(commands) do
			if param == cmd_name or param:sub(1, #cmd_name + 1) == cmd_name .. " " then
				return def.func(name, param:sub(#cmd_name + 2))
			end
		end

		local msg = {
			S("Usage:")
		}
		for cmd_name, def in conquer.orderedPairs(commands) do
			if def.params then
				msg[#msg + 1] = " - " .. cmd_name .. ":" .. def.params
			else
				msg[#msg + 1] = " - " .. cmd_name
			end
		end

		return false, table.concat(msg, "\n")
	end
})

conquer.register_command("start", {
	func = function(name)
		local player = minetest.get_player_by_name(name)
		if not player then
			return false, S("You need to be logged in!")
		end

		if conquer.get_session(player) then
			return false, S("You're already in a session")
		end

		local session = conquer.create_session(name, player:get_pos())
		local country = session:join(player)
		assert(country)

		return true, S("Started session as country @1", country.description)
	end,
})

conquer.register_command("leave", {
	func = function(name)
		local player = minetest.get_player_by_name(name)
		if not player then
			return false, S("You need to be logged in!")
		end

		local session = conquer.get_session(player)
		if not session then
			return false, S("You're not currently in a session")
		end

		session:leave(player)

		return true, S("Left session")
	end,
})

conquer.register_command("new_country", {
	func = function(name)
		local player = minetest.get_player_by_name(name)
		if not player then
			return false, S("You need to be logged in!")
		end

		local session = conquer.get_session(player)
		if not session then
			return false, S("There is no active session")
		end

		local country = session:create_country()
		if not country then
			return false, S("Unable to create country")
		end

		return true, S("Created new country: @1", country.description)
	end
})

conquer.register_command("join_country", {
	func = function(name, param)
		local player = minetest.get_player_by_name(name)
		if not player then
			return false, S("You need to be logged in!")
		end

		local session = conquer.get_session(player)
		if not session then
			return false, S("There is no active session")
		end

		if not session:set_player_country(player, param) then
			return false, S("Unable to set the country, does it exist?")
		end

		return true, S("Joined country @1", param:trim())
	end
})

conquer.register_command("new_join", {
	func = function(name)
		local player = minetest.get_player_by_name(name)
		if not player then
			return false, S("You need to be logged in!")
		end

		local session = conquer.get_session(player)
		if not session then
			return false, S("There is no active session")
		end

		local country = session:create_country()
		if not country then
			return false, S("Unable to create country")
		end

		assert(session:set_player_country(player, country.id))

		return true, S("Joined new country @1", country.description)
	end
})

conquer.register_command("join", {
	params = "",
	func = function(name, target_name)
		local player = minetest.get_player_by_name(name)
		if not player then
			return false, S("You need to be logged in!")
		end

		local target = minetest.get_player_by_name(target_name)
		if not target then
			return false, S("Player @1 is not online", target_name)
		end

		local dist = vector.distance(target:get_pos(), player:get_pos())
		if dist > 50 then
			return false, S("You need to be closer than 50 nodes")
		end

		local session = conquer.get_session(target)
		if not session then
			return false, S("Player @1 is not currently playing conquer", target_name)
		end

		if conquer.get_session(player) then
			return false, S("You're already in a session")
		end

		local country = session:join(player)
		assert(country)

		return true, S("Joined @1's session as country @2", target_name, country.description)
	end
})
