local S = conquer.S

local function on_interact(pos, _, player)
	local session, country = conquer.get_session_country(player)
	if not session or not country then
		return
	end

	local meta = minetest.get_meta(pos)
	if session.id ~= meta:get_string("conquer:session") or
			country.id ~= meta:get_string("conquer:country") then
		minetest.chat_send_player(player:get_player_name(),
			"This is not your keep")
		return
	end

	conquer.show_keep_gui(player, session, country)
end

conquer.register_building("keep", {
	description = S("Keep"),

	groups = { conquer = 1, not_in_creative_inventory = 1 },

	drawtype = "mesh",
	mesh = "conquer_keep.obj",

	health = 100,
	armor_groups = {
		conquer_blunt = 20,
		conquer_sharp = 0,
	},
	damage_levels = {
		{
			min_health = 80,
			tiles = {
				"conquer_keep.png"
			},
		},

		{
			min_health = 50,
			tiles = {
				"conquer_keep_damaged.png"
			},
		},

		{
			-- min_health = 10,
			tiles = {
				"conquer_keep_damaged.png"
			},
			mesh = "conquer_keep_damaged.obj",
		},

		-- {
		-- 	tiles = {
		-- 		"conquer_keep_damaged.png"
		-- 	},
		-- 	mesh = "conquer_keep_destroyed.obj",
		-- },
	},

	selection_box = {
		type = "fixed",
		fixed = {
			{ -6/16, -8/16, -6/16, 6/16, 6/16, 6/16 },
		},
	},

	collision_box = {
		type = "fixed",
		fixed = {
			{ -6/16, -8/16, -6/16, 6/16, 4/16, 6/16 },
		},
	},

	after_place_node = function(pos, player)
		local session, country = conquer.get_session_country(player)
		local success, message = conquer.place_keep(session, country, pos)
		if not success then
			minetest.chat_send_player(player:get_player_name(), message)
			minetest.set_node(pos, { name = "air" })
			return true
		end

		local meta = minetest.get_meta(pos)
		meta:set_string("conquer:session", session.id)
		meta:set_string("conquer:country", country.id)

		local inv = meta:get_inventory()
		inv:set_size("stockpile", 4)
		inv:add_item("stockpile", ItemStack("conquer:wood 99"))
		inv:add_item("stockpile", ItemStack("conquer:stone 99"))
		inv:add_item("stockpile", ItemStack("conquer:food 200"))
	end,

	on_punch = on_interact,
	on_rightclick = on_interact,

	allow_metadata_inventory_move = function()
		return 0
	end,

	allow_metadata_inventory_put = function()
		return 0
	end,

	allow_metadata_inventory_take = function()
		return 0
	end,
})

conquer.register_on_building_placed(function(session, country, building_type)
	if building_type ~= "keep" then
		return
	end

	local spawn_pos = conquer.find_empty_neighbour(country.keep) or
			vector.add(country.keep, vector.new(0, 1, 0))
	spawn_pos = vector.add(spawn_pos, vector.new(0, -0.49, 0))
	local obj = minetest.add_entity(spawn_pos, "conquer:unit")
	obj:get_luaentity():setup("knight", session, country)
end)


conquer.register_building("barracks", {
	description = S("Barracks"),

	groups = { conquer = 1, not_in_creative_inventory = 1 },

	drawtype = "mesh",
	tiles = {
		"conquer_barracks.png"
	},
	mesh = "conquer_barracks.obj",

	health = 100,
	armor_groups = {
		conquer_blunt = 100,
		conquer_sharp = 20,
	},
	damage_levels = {
		{
			min_health = 80,
			tiles = {
				"conquer_barracks.png"
			},
		},

		{
			min_health = 50,
			tiles = {
				"conquer_barracks_damaged.png"
			},
		},

		{
			min_health = 10,
			tiles = {
				"conquer_barracks_damaged.png"
			},
			mesh = "conquer_barracks_damaged.obj",
		},

		{
			tiles = {
				"conquer_barracks_damaged.png"
			},
			mesh = "conquer_barracks_destroyed.obj",
		},
	},

	selection_box = {
		type = "fixed",
		fixed = {
			{ -5/16, -8/16, -5/16, 5/16, -1/16, 5/16 },
		},
	},

	collision_box = {
		type = "fixed",
		fixed = {
			{ -5/16, -8/16, -5/16, 5/16, -1/16, 5/16 },
		},
	},

	after_place_node = function(pos, player)
		local session, country = conquer.get_session_country(player)
		local success, message = conquer.place_barracks(session, country, pos)
		if not success then
			minetest.chat_send_player(player:get_player_name(), message)
			minetest.set_node(pos, { name = "air" })
			return true
		end

		local meta = minetest.get_meta(pos)
		meta:set_string("conquer:session", session.id)
		meta:set_string("conquer:country", country.id)
	end,

	on_timer = function(pos)
		local meta = minetest.get_meta(pos)

		local session = conquer.get_session_by_id(meta:get_string("conquer:session"))
		if not session then
			return false
		end

		local country = session:get_country_by_id(meta:get_string("conquer:country"))
		if not country then
			return false
		end

		local unit_type_name = meta:get("conquer:unit_type")
		if not unit_type_name then
			return false
		end

		meta:set_string("conquer:unit_type", "")

		local spawn_pos = conquer.find_empty_neighbour(country.barracks) or
			vector.add(country.barracks, vector.new(0, 1, 0))
		spawn_pos = vector.add(spawn_pos, vector.new(0, -0.49, 0))
		local obj = minetest.add_entity(spawn_pos, "conquer:unit")
		obj:get_luaentity():setup(unit_type_name, session, country)

		return false
	end,
})

local function get_stockpile_by_country(country)
	if not country.keep then
		return nil
	end

	local inv = minetest.get_inventory({
		type = "node",
		pos = country.keep
	})
	if not inv then
		return nil
	end

	return inv
end

conquer.register_on_can_train_unit(function(_, country, unit_type)
	if not country.barracks then
		return S("You need a barracks")
	end

	if conquer.get_building_hp(country.barracks) < 50 then
		return S("Unable to train whilst the barracks are damaged")
	end

	local inv = get_stockpile_by_country(country)
	if not inv then
		return S("Unable to find stockpile")
	end

	local meta = minetest.get_meta(country.barracks)
	if meta:contains("conquer:unit_type") then
		return S("A unit is already training")
	end

	local craft = unit_type.craft
	for _, item in pairs(craft.resources or {}) do
		if not inv:contains_item("stockpile", item) then
			return S("Missing item @1" , ItemStack(item):get_description())
		end
	end
end)

conquer.register_on_train_unit(function(_, country, unit_type)
	assert(country.barracks)

	local inv = get_stockpile_by_country(country)
	assert(inv)

	local craft = unit_type.craft
	for _, item in pairs(craft.resources or {}) do
		assert(inv:contains_item("stockpile", item))
		inv:remove_item("stockpile", item)
	end

	assert(country.barracks)

	local meta = minetest.get_meta(country.barracks)
	assert(not meta:contains("conquer:unit_type"))

	meta:set_string("conquer:unit_type", unit_type.name)

	local timer = minetest.get_node_timer(country.barracks)
	timer:start(unit_type.craft.time)
end)
