local _unit_types = {}
local _sessions = {}
local _selected_units = conquer.make_context_provider()

local function make_callback(name)
	local callbacks = {}
	conquer["register_on_" .. name] = function(func)
		table.insert(callbacks, func)
	end

	conquer["run_on_" .. name] = function(...)
		for i=1, #callbacks do
			callbacks[i](...)
		end
	end
end

local function make_returning_callback(name)
	local callbacks = {}
	conquer["register_on_" .. name] = function(func)
		table.insert(callbacks, func)
	end

	conquer["run_on_" .. name] = function(...)
		for i=1, #callbacks do
			local ret = callbacks[i](...)
			if ret ~= nil then
				return ret
			end
		end
	end
end

make_callback("building_placed")
make_callback("selected_units_changed")
make_callback("join_session")
make_callback("leave_session")
make_callback("change_country")
make_callback("train_unit")
make_returning_callback("can_train_unit")
make_callback("unit_created")
make_callback("building_destroyed")

local S = conquer.S

function conquer.create_session(owner, pos)
	local session = conquer.Session:new(owner, pos)
	_sessions[session.id] = session
	return session
end

function conquer.import_session(data)
	local session = conquer.Session:new(data)
	_sessions[session.id] = session
	return session
end

function conquer.delete_session(session)
	assert(#conquer.get_players_in_session(session) == 0)
	_sessions[session.id] = nil
end

function conquer.get_session_by_id(pos)
	assert(type(pos) == "string")
	return _sessions[pos]
end

function conquer.get_sessions()
	return _sessions
end

function conquer.place_keep(session, country, pos)
	if not session or not country then
		return false, S("You need to join a country")
	end

	assert(session.get_countries)
	assert(country.colors.primary)

	if country.keep ~= nil then
		return false, S("You've already placed a keep")
	end

	for _, other_country in pairs(session:get_countries()) do
		if other_country.keep and vector.distance(other_country.keep, pos) < 15 then
			return false, S("You're two close to @1's keep! You need to be at least @2 nodes away",
					other_country.description, 15)
		end
	end

	country.keep = vector.new(pos)

	conquer.run_on_building_placed(session, country, "keep")

	return true
end

function conquer.place_building(building_type, session, country, pos)
	local singleton = building_type == "barracks"

	if not session or not country then
		return false, S("You need to join a country")
	end

	assert(session.get_countries)
	assert(country.colors.primary)

	if country.keep == nil then
		return false, S("You need to place a keep")
	end

	if singleton and country[building_type] ~= nil then
		return false, S("This building may only be placed once")
	end

	if vector.distance(country.keep, pos) > 5 then
		return false, S("Buildings must be placed within 5 nodes of your keep")
	end

	if singleton then
		country[building_type] = vector.new(pos)
	end

	conquer.run_on_building_placed(session, country, building_type)

	return true
end

function conquer.get_players_in_session(session)
	local ret = {}
	for _, player in pairs(minetest.get_connected_players()) do
		if player:get_meta():get("conquer:session") == session.id then
			ret[#ret + 1] = player
		end
	end

	return ret
end

function conquer.get_session(player)
	assert(player.get_player_name, "get_session expects a player")

	local session_id = player:get_meta():get("conquer:session")
	return session_id and _sessions[session_id]
end

function conquer.get_session_country(player)
	assert(player.get_player_name, "get_session_country expects a player")

	local session = conquer.get_session(player)
	if not session then
		return nil, nil
	end

	local country = session:get_country_by_player(player:get_player_name())
	if not country then
		return nil, nil
	end

	return session, country
end

local function get_selected_units_raw(player)
	assert(player.get_player_name, "get_selected_units expects a player")

	local session, country = conquer.get_session_country(player)
	local context = _selected_units(player)
	if not country or not context.selected_units then
		return {}, false, nil, nil
	end

	local changed = false
	local units = context.selected_units

	local pointer=1
	for i=1, #units do
		if units[i]:get_luaentity() ~= nil then
			units[pointer] = units[i]
			pointer = pointer + 1
		end
	end
	for i=pointer, #units do
		changed = true
		units[i] = nil
	end

	return units, changed, session, country
end

function conquer.get_selected_units(player)
	local units, changed, session, country = get_selected_units_raw(player)

	if changed then
		conquer.run_on_selected_units_changed(player, session, country, units)
	end

	return units
end

function conquer.select_unit(player, obj)
	assert(player.get_player_name, "select_unit expects a player")
	assert(obj.get_pos and obj.get_luaentity)

	local session, country = conquer.get_session_country(player)
	if not session then
		return false
	end

	local entity = obj:get_luaentity()
	if not entity then
		return false
	end

	if not entity.select or not entity:select(session, country) then
		return false
	end

	local context = _selected_units(player)
	context.selected_units = context.selected_units or {}
	table.insert(context.selected_units, obj)

	conquer.run_on_selected_units_changed(player, session, country, get_selected_units_raw(player))

	return true
end

function conquer.deselect_unit(player, unit)
	assert(player.get_player_name, "deselect_unit expects a player")
	assert(unit.get_luaentity, "deselect_unit expects a unit")

	if not unit:get_luaentity() then
		return false
	end

	local session, country = conquer.get_session_country(player)
	if not session then
		return false
	end

	local context = _selected_units(player)
	local units = context.selected_units or {}

	for i=1, #units do
		if units[i] == unit then
			table.remove(units, i)
			unit:get_luaentity():deselect()

			conquer.run_on_selected_units_changed(player, session, country, get_selected_units_raw(player))
			return true
		end
	end

	return false
end

function conquer.deselect_all_units(player)
	assert(player.get_player_name, "deselect_all_units sexpects a player")

	local session, country = conquer.get_session_country(player)
	if not session then
		return false
	end

	local context = _selected_units(player)
	local units = context.selected_units or {}
	for i=#units, 1, -1 do
		if units[i]:get_luaentity() then
			units[i]:get_luaentity():deselect()
		end
		units[i] = nil
	end

	assert(#units == 0 and next(units) == nil)

	conquer.run_on_selected_units_changed(player, session, country, units)

	return true
end

function conquer.replace_select_unit(player, obj)
	assert(player.get_player_name, "deselect_all_units sexpects a player")

	local session, country = conquer.get_session_country(player)
	if not session then
		return false
	end

	local context = _selected_units(player)
	local units = context.selected_units or {}
	context.selected_units = units

	for i=#units, 1, -1 do
		if units[i]:get_luaentity() then
			units[i]:get_luaentity():deselect()
		end
		units[i] = nil
	end

	local entity = obj:get_luaentity()
	if not entity then
		return false
	end

	if not entity.select or not entity:select(session, country) then
		return false
	end

	table.insert(units, obj)
	assert(#units == 1)

	conquer.run_on_selected_units_changed(player, session, country, units)

	return true
end

function conquer.register_unit(name, def)
	assert(not _unit_types[name], "Unit type already registered!")
	assert(def.get_properties, "get_properties is required in unit type")
	assert(def.craft, "def.craft is required, as a table with at least time")
	assert(def.craft.time, "craft time is required")

	assert(not def.melee or def.melee.tool_capabilities)
	assert(not def.ranged or type(def.ranged.min_range) == "number")
	assert(not def.ranged or type(def.ranged.max_range) == "number")
	assert(not def.ranged or def.ranged.max_range > def.ranged.min_range)
	assert(not def.ranged or def.ranged.tool_capabilities)

	def.name = name
	_unit_types[name] = def
end

function conquer.get_unit_type(name)
	assert(type(name) == "string")

	return _unit_types[name]
end

function conquer.get_unit_types()
	return _unit_types
end

function conquer.can_train_unit(session, country, unit_type)
	local message = conquer.run_on_can_train_unit(session, country, unit_type)
	if message then
		return false, message
	end

	return true, nil
end

function conquer.train_unit(session, country, unit_type)
	conquer.run_on_train_unit(session, country, unit_type)
end
