grouplib = {}
local override_queue = {}

local registered_items = minetest.registered_items

-- Gets the groups of an item by name.
-- Either returns a table or nil.
--
--     groups = grouplib.item_get("sand")
--
function grouplib.item_get(name)
	if not name then return end

	assert(type(name) == "string", "Name passed to grouplib.item_get should be a string")

	local def = registered_items[name]
	if not def then
		minetest.log("warning", "[grouplib] No definition found for "..name..", returning early")
		return
	end

	return def.groups
end

-- Sets the groups of an item by name.
-- This is not the same as just plain overriding, because this doesn't replace
-- the entire groups table of an item, only update it with new values.
--
--     grouplib.item_set("sand", {yellow = 1, crumbly = -1, magic = 9})
--
function grouplib.item_set(name, groups)
	if not name or not groups then return end

	assert(type(name) == "string", "Name passed to grouplib.item_set should be a string")
	assert(type(groups) == "table", "Groups passed to grouplib.item_set should be a table")

	local def = registered_items[name]
	if not def then
		minetest.log("warning", "[grouplib] No definition found for "..name..", returning early")
		return
	end

	def.groups = def.groups or {}
	for k, v in pairs(groups) do
		def.groups[k] = v
	end
end

-- Replaces the groups of an item by name.
-- The old groups are completely thrown out; if what you want to do is to update
-- the existing values then use `grouplib.item_set`.
--
--     grouplib.item_replace("sand", {oddly_breakable_by_hand = 1})
--
function grouplib.item_replace(name, groups)
	if not name or not groups then return end

	assert(type(name) == "string", "Name passed to grouplib.item_set should be a string")
	assert(type(groups) == "table", "Groups passed to grouplib.item_set should be a table")

	local def = registered_items[name]
	if not def then
		minetest.log("warning", "[grouplib] No definition found for "..name..", returning early")
		return
	end

	def.groups = def.groups or {}
	for k, v in pairs(groups) do
		def.groups[k] = v
	end
end

-- Gets the ratings of groups of an item by its name and group names.
-- The amount of return values will be the same as the amount of secondary
-- arguments passed to the function.
--
-- one, two, three = grouplib.item_get_rating("sand", "crumbly", "falling_node", "soil")
--
function grouplib.item_get_rating(name, ...)
	if not name or not ... then return end

	assert(type(name) == "string", "Name passed to grouplib.item_get_rating should be a string")

	local def = registered_items[name]
	if not def then
		minetest.log("warning", "[grouplib] No definition found for "..name..", returning early")
		return
	end

	local ret = {}
	local arg = {...}
	for i = 1, #arg do
		ret[#ret+1] = def.groups[arg[i]] or 0
	end

	return unpack(ret)
end

-- Bulk-overrides all items with specified group (~= 0) with the override table.
-- Values are added to a queue on call and the overrides are performed after all
-- mods have loaded.
-- Keep in mind this is basically a wrapper for `minetest.override_item`.
--
--     grouplib.group_override("leaves", {drawtype = "firelike"})
--
function grouplib.group_override(group, override)
	if not override_queue[group] then
		override_queue[group] = {}
	end
	local queue = override_queue[group]
	queue[#queue+1] = override
end

minetest.register_on_mods_loaded(function()
	for name, def in pairs(registered_items) do
		local groups = def.groups
		for group, overrides in pairs(override_queue) do
			if groups[group] and groups[group] ~= 0 then
				for i = 1, #overrides do
					minetest.override_item(name, overrides[i])
				end
			end
		end
	end
end)
