-- mcl_decor/src/fridge.lua

local S = ...
local F = core.formspec_escape
local C = core.colorize

local sf = string.format

local open_fridges = {}

local drop_content = mcl_util.drop_items_from_meta_container("main")

-- BEGIN Sorter functions borrowed from mcl_chest
local function double_chest_add_item(top_inv, bottom_inv, listname, stack)
	if not stack or stack:is_empty() then
		return
	end

	local name = stack:get_name()

	local function top_off(inv, stack)
		for c, chest_stack in ipairs(inv:get_list(listname)) do
			if stack:is_empty() then
				break
			end

			if chest_stack:get_name() == name and chest_stack:get_free_space() > 0 then
				stack = chest_stack:add_item(stack)
				inv:set_stack(listname, c, chest_stack)
			end
		end

		return stack
	end

	stack = top_off(top_inv, stack)
	stack = top_off(bottom_inv, stack)

	if not stack:is_empty() then
		stack = top_inv:add_item(listname, stack)
		if not stack:is_empty() then
			bottom_inv:add_item(listname, stack)
		end
	end
end

local function limit_put_list(stack, list)
	for _, other in ipairs(list) do
		stack = other:add_item(stack)
		if stack:is_empty() then
			break
		end
	end
	return stack
end

local function limit_put(stack, inv1, inv2)
	local leftover = ItemStack(stack)
	leftover = limit_put_list(leftover, inv1:get_list("main"))
	leftover = limit_put_list(leftover, inv2:get_list("main"))
	return stack:get_count() - leftover:get_count()
end
-- END Sorter functions borrowed from mcl_chest

local function on_blast(pos)
	local node = core.get_node(pos)
	drop_content(pos, node)
	core.remove_node(pos)

	local above = vector.new(pos.x, pos.y+1, pos.z)
	core.remove_node(above)
end

-- Simple protection checking functions
local function protection_check_move(pos, from_list, from_index, to_list, to_index, count, player)
	local name = player:get_player_name()
	if core.is_protected(pos, name) then
		core.record_protection_violation(pos, name)
		return 0
	else
		return count
	end
end

local function protection_check_take(pos, listname, index, stack, player)
	local name = player:get_player_name()
	if core.is_protected(pos, name) then
		core.record_protection_violation(pos, name)
		return 0
	else
		return stack:get_count()
	end
end

local function protection_check_put(pos, listname, _, stack, player)
	if core.get_item_group(stack:get_name(), "food") == 0 then
		return 0
	end
	local other_pos = vector.offset(pos, 0, 1, 0)
	local name = player:get_player_name()
	if core.is_protected(pos, name) then
		core.record_protection_violation(pos, name)
		return 0
		-- BEGIN OF LISTRING WORKAROUND
	elseif listname == "input" then
		local other_inv = core.get_inventory({ type = "node", pos = other_pos })
		local inv = core.get_inventory({ type = "node", pos = pos })
		return limit_put(stack, other_inv, inv)
		-- END OF LISTRING WORKAROUND
	else
		return stack:get_count()
	end
end

local function fridge_open(pos, node, clicker)
	local name = core.get_meta(pos):get_string("name")

	if name == "" then
		name = S("Fridge")
	end

	local meta = core.get_meta(pos)
	local meta_top = core.get_meta(vector.offset(pos, 0, 1, 0))
	local inv = meta:get_inventory()
	local inv_top = meta_top:get_inventory()
	if inv:get_size("input") == 0 and inv_top:get_size("main") == 0 then
		inv:set_size("input", 1)
		inv_top:set_size("main", 9 * 3)
	end

	local playername = clicker:get_player_name()

	core.show_formspec(playername,
		"mcl_decor:fridge_" .. pos.x .. "_" .. pos.y .. "_" .. pos.z,
		table.concat({
			"formspec_version[4]",
			"size[11.75,14.15]",

			"label[0.375,0.375;" .. F(C(mcl_formspec.label_color, name)) .. "]",
			mcl_formspec.get_itemslot_bg_v4(0.375, 0.75, 9, 3),
			sf("list[nodemeta:%s,%s,%s;main;0.375,0.75;9,3;]", pos.x, pos.y + 1, pos.z),
			mcl_formspec.get_itemslot_bg_v4(0.375, 4.5, 9, 3),
			sf("list[nodemeta:%s,%s,%s;main;0.375,4.5;9,3;]", pos.x, pos.y, pos.z),
			"label[0.375,8.45;" .. F(C(mcl_formspec.label_color, S("Inventory"))) .. "]",
			mcl_formspec.get_itemslot_bg_v4(0.375, 8.825, 9, 3),
			"list[current_player;main;0.375,8.825;9,3;9]",

			mcl_formspec.get_itemslot_bg_v4(0.375, 12.775, 9, 1),
			"list[current_player;main;0.375,12.775;9,1;]",

			--BEGIN OF LISTRING WORKAROUND
			"listring[current_player;main]",
			sf("listring[nodemeta:%s,%s,%s;input]", pos.x, pos.y, pos.z),
			--END OF LISTRING WORKAROUND
			"listring[current_player;main]",
			sf("listring[nodemeta:%s,%s,%s;main]",pos.x, pos.y + 1, pos.z),
			"listring[current_player;main]",
			sf("listring[nodemeta:%s,%s,%s;main]",pos.x, pos.y, pos.z),
		})
	)

	core.swap_node(pos, { name = "mcl_decor:fridge_open", param2 = node.param2 })
	open_fridges[playername] = pos
	core.sound_play({name="mcl_decor_fridge_open", gain=0.5}, {
		pos = pos,
		max_hear_distance = 16,
	}, true)
end

local function close_forms(pos)
	local formname = "mcl_decor:fridge_" .. pos.x .. "_" .. pos.y .. "_" .. pos.z
	local players = core.get_connected_players()
	for p = 1, #players do
		if vector.distance(players[p]:get_pos(), pos) <= 30 then
			core.close_formspec(players[p]:get_player_name(), formname)
		end
	end
end

local function update_after_close(pos)
	local node = core.get_node_or_nil(pos)
	if not node then return end
	if node.name == "mcl_decor:fridge_open" then
		core.swap_node(pos, { name = "mcl_decor:fridge", param2 = node.param2 })
		core.sound_play({name="mcl_decor_fridge_close", gain=0.5}, {
			pos = pos,
			max_hear_distance = 16,
		}, true)
	end
end

local function close_fridge(player)
	local name = player:get_player_name()
	local open = open_fridges[name]
	if open == nil then
		return
	end

	update_after_close(open)

	open_fridges[name] = nil
end


-- Reused constants
local FRIDGE_BOX = {
	type = "fixed",
	fixed = {
		{-8/16, -8/16, -5/16, 8/16, 24/16, 8/16},
	}
}

-- Dummy node
core.register_node("mcl_decor:fridge_top", {
	_doc_items_create_entry = false,
	drawtype = "airlike",
	tiles = {"blank.png"},
	groups = {not_in_creative_inventory = 1},
	use_texture_alpha = "clip",
	is_ground_content = false,
	pointable = false,
	collision_box = {
		type = "fixed",
		fixed = {
			{-8/16, -8/16, -5/16, 8/16, 8/16, 8/16},
		}
	},
	sunlight_propagates = true,
	paramtype = "light",
	on_construct = function(pos)
		local meta = core.get_meta(pos)
		local inv = meta:get_inventory()
		inv:set_size("main", 9 * 3)
	end,
	on_destruct = drop_content,
	allow_metadata_inventory_put = protection_check_put
})

local closed_def = {
	description = S("Fridge"),
	_tt_help = S("54 inventory slots"),
	_doc_items_longdesc = S("Fridges are containers which provide 54 inventory slots."),
	_doc_items_usagehelp = S("To access its inventory, rightclick it. When broken, the items will drop out."),
	tiles = {"mcl_decor_fridge.png"},
	inventory_image = "mcl_decor_fridge_inv.png",
	drawtype = "mesh",
	mesh = "mcl_decor_fridge.obj",
	paramtype = "light",
	paramtype2 = "4dir",
	is_ground_content = false,
	selection_box = FRIDGE_BOX,
	collision_box = FRIDGE_BOX,
	groups = {pickaxey=2, deco_block=1},
	sounds = mcl_sounds.node_sound_metal_defaults(),
	_mcl_blast_resistance = 6,
	_mcl_hardness = 5,
	on_construct = function(pos)
		local meta = core.get_meta(pos)
		local inv = meta:get_inventory()
		inv:set_size("main", 9 * 3)
		--[[ The "input" list is *another* workaround (hahahaha!) around the fact that Minetest
		does not support listrings to put items into an alternative list if the first one
		happens to be full. See <https://github.com/minetest/minetest/issues/5343>.
		This list is a hidden input-only list and immediately puts items into the appropriate chest.
		It is only used for listrings and hoppers. This workaround is not that bad because it only
		requires a simple “inventory allows” check for large chests.]]
		-- FIXME: Refactor the listrings as soon Minetest supports alternative listrings
		-- BEGIN OF LISTRING WORKAROUND
		inv:set_size("input", 1)
		-- END OF LISTRING WORKAROUND
	end,
	on_place = function(itemstack, placer, pointed_thing)
		-- Use pointed node's on_rightclick function first, if present
		local rightclick = mcl_util.call_on_rightclick(itemstack, placer, pointed_thing)
		if rightclick then
			return rightclick
		end

		local above = core.get_node_or_nil(vector.offset(pointed_thing.above, 0, 1, 0))
		-- Don't place fridge if no space available
		if not above or above.name ~= "air" then
			return itemstack
		end

		-- Finally, if all checks pass
		return core.item_place(itemstack, placer, pointed_thing)
	end,
	after_place_node = function(pos, placer, itemstack, pointed_thing)
		core.get_meta(pos):set_string("name", itemstack:get_meta():get_string("name"))
		local above = vector.new(pos.x, pos.y+1, pos.z)
		-- Place the dummy node
		core.set_node(above, {name="mcl_decor:fridge_top"})
	end,
	after_destruct = function(pos)
		-- Remove the dummy node
		core.remove_node(vector.new(pos.x, pos.y+1, pos.z))
	end,
	allow_metadata_inventory_move = protection_check_move,
	allow_metadata_inventory_take = protection_check_take,
	allow_metadata_inventory_put = protection_check_put,
	on_metadata_inventory_move = function(pos, from_list, from_index, to_list, to_index, count, player)
		core.log("action", player:get_player_name() ..
			" moves stuff in fridge at " .. core.pos_to_string(pos))
	end,
	on_metadata_inventory_put = function(pos, listname, _, stack, player)
		core.log("action", player:get_player_name() ..
			" moves stuff to fridge at " .. core.pos_to_string(pos))
		local other_pos = vector.offset(pos, 0, 1, 0)
		-- BEGIN OF LISTRING WORKAROUND
		if listname == "input" then
			local other_inv = core.get_inventory({ type = "node", pos = other_pos })
			local inv = core.get_inventory({ type = "node", pos = pos })

			inv:set_stack("input", 1, nil)

			double_chest_add_item(other_inv, inv, "main", stack)
		end
		-- END OF LISTRING WORKAROUND
	end,
	on_metadata_inventory_take = function(pos, listname, index, stack, player)
		core.log("action", player:get_player_name() ..
			" takes stuff from fridge at " .. core.pos_to_string(pos))
	end,
	after_dig_node = drop_content,
	on_blast = on_blast,
	on_rightclick = fridge_open,
	on_destruct = close_forms,
}

local open_def = table.copy(closed_def)
open_def._doc_items_create_entry = false
open_def.groups.not_in_creative_inventory = 1
open_def.drop = "mcl_decor:fridge"

core.register_node("mcl_decor:fridge", closed_def)
core.register_node("mcl_decor:fridge_open", open_def)

core.register_craft({
	output = "mcl_decor:fridge",
	recipe = {
		{"mcl_core:iron_ingot", "mcl_core:iron_ingot", "mcl_core:iron_ingot"},
		{"mcl_nether:quartz", "group:ice", "mcl_nether:quartz"},
		{"mcl_nether:quartz", "mcl_core:iron_ingot", "mcl_nether:quartz"},
	}
})

core.register_on_player_receive_fields(function(player, formname, fields)
	if formname:find("mcl_decor:fridge") == 1 and fields.quit then
		close_fridge(player)
	end
end)

core.register_on_leaveplayer(function(player)
	close_fridge(player)
end)
