local xyz = {"x","y","z"}
local S = core.get_translator("traverse")
local beam_speed = 0.2

local corner_two = vector.new(0.5,0.499,0.5)
local corner_one = vector.new(-0.5,-0.5,-0.5)
local gravity = vector.new(0,-9.81,0)

local has_digilines = core.get_modpath("digilines") and true

function remove_mover_if_closer(o,pos) --returns first child of mover or non-mover entity or nil
	local e = o:get_luaentity()
	if e and e.name == "traverse:mover" then
		if (vector.distance(o:get_pos(),pos) < vector.distance(o:get_pos(),e._origin)) then
			local child = o:get_children()[1]
			o:remove()
			return child
		end
	else
		return o
	end
end

core.register_entity("traverse:mover", {
	initial_properties = {
		hp_max = 1,
		physical = true,
		collide_with_objects = false,
		collisionbox = {-0.3, -0.3, -0.3, 0.3, 0.3, 0.3},
		textures = {"traverse_transparent.png"},
		static_save = false, --unless properties are serialized (and serializable, attached children aren't,) there is no point in saving the entity
		pointable = false,
		--~ is_visible = false, --buggy, causes child position to not be updated properly

	},
	on_step = function (self,dt,moveresult)
		if moveresult and moveresult.collides then
			self.object:remove()
			return
		end
		
		if self._mode == "beam" then
			local p	 = self.object:get_pos()
			if not vector.in_area(p, self._area[1], self._area[2]) then
				self.object:remove()
				return
			end
			
			self._time = self._time - dt
			if self._time < 0 then
				self._time = self._time + beam_speed
				--check that you're in the traction area and update values from tactor beam like pull and that the beam isn't off
				local node = core.get_node(self._origin)
				
				--if beam is off, moved or destroyed, destroy mover
				if node.name ~= "traverse:beam" then
					self.object:remove()
					return
				end
				local meta = core.get_meta(self._origin)
				local pull = meta:get_int("pull")==1
				if pull ~= self._pull then
					self.object:remove()
					return
				end
			end
		elseif self._mode == "cannon" then
			local o = self.object
			o:set_acceleration(gravity)
		end
	end,
	--can't serialize attached object, so this is pointless for now
	--~ get_staticdata = function (self)
		--~ return core.write_json({
			--~ mode = self._mode,
			--~ origin = self._origin,
			--~ vel = self:get_velocity(),
		--~ })
	--~ end,
	on_activate =  function (self, staticdata, dtime_s)
		if staticdata ~= "" and staticdata ~= nil then
			local data = core.parse_json(staticdata) or {}
			self._origin = data.origin
			self._mode = data.mode
		end
		if self._mode == "beam" then
			self._time = beam_speed
			local meta = core.get_meta(self._origin)
			self._pull = meta:get_int("pull")==1
			
		end
	end,
	
	on_deactivate = function (self)
		if self._audio then
			core.sound_stop(self._audio)
		end
	end,
})

local function is_movable(o)
	if not o:get_attach() then
		local e = o:get_luaentity()
		if o:is_player() or e.name == "__builtin:item" or e.is_mob or e._creatura_mob or e._cmi_is_mob then --mcl, creatura, mobs
			return true
		end
	end
end
local function bool2int(b) return b=="true" and 1 or 0 end
local function int2bool(i) return i==1 and "true" or "false" end
local player_data = {}

local function swap_beam_if_changed(node,pos,to_off)
	local name = node.name
	if name == "traverse:beam" and to_off then 
		node.name="traverse:beam_off"
		core.swap_node(pos,node)
	elseif name == "traverse:beam_off" and not to_off then
		node.name="traverse:beam"
		core.swap_node(pos,node)
		core.get_node_timer(pos):start(0.1)
	end
end
local function get_cannon_entity(pos)
	local objects = core.get_objects_inside_radius(pos, 0.5)
	local found
	for _, obj in ipairs(objects) do
		if obj and obj:get_luaentity() and obj:get_luaentity().name == "traverse:cannon" then
			found = obj
			break
		end
	end
	return found
end

local function vector_to_yaw(dir)
	return math.atan2(dir.z, dir.x)
end

local function vector_to_pitch(dir)
	return math.atan2(dir.y, math.sqrt(dir.x * dir.x + dir.z * dir.z))
end
local function set_entity_rotation(entity, dir)
	local yaw = vector_to_yaw(dir)
	local pitch = vector_to_pitch(dir)
	entity:set_rotation(vector.new(0,yaw,-pitch))
end

local function adjust_cannon(pos,meta,x,y,z)
	if type(x) ~= "number" then return end
	if type(y) ~= "number" then return end
	if type(z) ~= "number" then return end
	meta:set_float("vel_x",x)
	meta:set_float("vel_y",y)
	meta:set_float("vel_z",z)
	local e = get_cannon_entity(pos)
	if e then
		set_entity_rotation(e,vector.new(x,y,z))
	end
	
end

local on_digiline_receive = function (pos, _, channel, msg)
	local node = core.get_node(pos)
	local name = node.name
	local meta = core.get_meta(pos)
	if channel == meta:get_string("channel") then
		if type(msg) == "table" then
			if name == "traverse:beam_off" or name == "traverse:beam" then
				if msg.pull ~= nil and type(msg.pull) == "boolean" then
					meta:set_int("pull",msg.pull and 1 or 0)
				end
				if msg.off ~= nil and type(msg.off) == "boolean" then
					swap_beam_if_changed(node,pos,msg.off)
				end
				if msg.reach ~= nil and type(msg.reach) == "number" then
					meta:set_int("reach",msg.reach)
				end
			elseif name == "traverse:cannon" then
				adjust_cannon(pos,meta,msg[1],msg[2],msg[3])
			end
		end
	end
end

local digilines =
	{
		receptor = {},
		effector = {
			action = on_digiline_receive
		},
	}

local configure_node = function(pos, node, clicker, itemstack, pointed_thing)
	local node = core.get_node(pos)
	local name = node.name
	local meta = core.get_meta(pos)
	local player_name = clicker:get_player_name()
	player_data[player_name] = pos
	if name == "traverse:beam" or name == "traverse:beam_off" then
		local str = meta:get_int("reach")
		core.show_formspec(player_name, "traverse:form",
		"formspec_version[5]size[4,3]" ..
		"checkbox[0.5,1.2;beam_pull;"..S("Pull")..";"..int2bool(meta:get_int("pull")).."]"..
		"checkbox[0.5,1.6;beam_off;"..S("Off")..";"..(name=="traverse:beam" and "false" or "true").."]"..
		"field[0.5,0.4;3,0.5;beam_reach;"..S("Reach")..";"..str.."]"
		..(has_digilines and "field[0.5,2;3,0.5;channel;"..S("Channel")..";"..meta:get_string("channel").."]" or ""))
	elseif name == "traverse:cannon" then
		local x = meta:get_float("vel_x")
		local y = meta:get_float("vel_y")
		local z = meta:get_float("vel_z")
		local str = x..","..y..","..z
		local str2 = pos.x..","..pos.y..","..pos.z
		core.show_formspec(player_name, "traverse:form",
		"formspec_version[5]size[4,3]" ..
		"field[0.5,0.5;3,0.5;cannon_velocity;"..S("Velocity Vector")..";"..str.."]"..
		--~ "field[0.5,1.5;3,0.5;cannon_target;"..S("Target Vector")..";"..str2.."]"..
		(has_digilines and "field[0.5,2;3,0.5;channel;"..S("Channel")..";"..meta:get_string("channel").."]" or ""))
	end
end

local function calculate_cannon_velocity_with_apex(cannon_pos, target_pos, apex_height) --doesn't work yet
	--testing coords: 136,46,352
	local g = 9.8

	-- Calculate differences in coordinates
	local dx = target_pos.x - cannon_pos.x
	local dz = target_pos.z - cannon_pos.z
	local dy = target_pos.y - cannon_pos.y

	-- Horizontal distance
	local horizontal_distance = math.sqrt(dx * dx + dz * dz)

	-- Vertical velocity needed to reach the apex height above the target
	local vy_apex = math.sqrt(2 * g * apex_height)

	-- Time to reach apex height
	local time_to_apex = vy_apex / g

	-- Total vertical distance to travel (ascent to apex + descent from apex to target)
	local total_vertical_distance = apex_height + dy

	-- Time to descend from apex to target
	local time_to_descend = math.sqrt((2 * total_vertical_distance) / g)

	-- Total time of flight
	local total_time = time_to_apex + time_to_descend

	-- Angle to target on the horizontal plane
	local theta = math.atan2(dz, dx)

	-- Horizontal velocity components
	local vh = horizontal_distance / total_time
	local vx = vh * math.cos(theta)
	local vz = vh * math.sin(theta)

	-- Initial vertical velocity component (upward to reach apex)
	local vy = vy_apex

	return vector.new(vx, vy, vz)
end



local function parse_vector(text)
	local xyz = string.split(text,",")
	return vector.new(tonumber(xyz[1]),tonumber(xyz[2]),tonumber(xyz[3]))
end
core.register_on_player_receive_fields(function(player, formname, fields)
	if formname == "traverse:form" then
		local player_name = player:get_player_name()
		local pos = player_data[player_name]
		local meta = core.get_meta(pos)
		
		if fields.cannon_velocity then
			--fix old cannons from < 2.0
			if not get_cannon_entity(pos) then
				local x = meta:get_float("vel_x")
				local y = meta:get_float("vel_y")
				local z = meta:get_float("vel_z")
				core.set_node(pos,{name="traverse:cannon"})
				meta = core.get_meta(pos)
				adjust_cannon(pos,meta,x,y,z)
			end
			
			local xyz
			if fields.key_enter_field == "cannon_target" then
				local target = parse_vector(fields.cannon_target)
				xyz = calculate_cannon_velocity_with_apex(pos, target, 5)
			else
				xyz = parse_vector(fields.cannon_velocity)
			end

			adjust_cannon(pos,meta,xyz[1],xyz[2],xyz[3])
			return
		end
		
		if fields.beam_reach then
			if tonumber(fields.beam_reach) then
				meta:set_int("reach",tonumber(fields.beam_reach))
			end
		end
		if fields.beam_pull then
			meta:set_int("pull",bool2int(fields.beam_pull))
		end
		if fields.beam_off then
			local node = core.get_node(pos)
			swap_beam_if_changed(node,pos,core.is_yes(fields.beam_off))
		end
		if (fields.channel) then
			core.get_meta(pos):set_string("channel", fields.channel)
		end
		if fields.quit then
			player_data[player_name] = nil
		end
	end
end)

core.register_node("traverse:beam_off", {
	digilines = digilines,
	description = S("Tractor Beam Generator"),
	drop = "traverse:beam",
	tiles = {"traverse_cannon.png^[hsl:30","traverse_cannon.png^[hsl:30","traverse_cannon.png^[hsl:30","traverse_cannon.png^[hsl:30","traverse_cannon.png^[hsl:30","traverse_cannon.png^[hsl:30^traverse_beam_front.png"},
	groups = {cracky = 2,not_in_creative_inventory = 1},
	paramtype2 = "facedir",
	on_rightclick = configure_node,
})
core.register_node("traverse:beam", {
	digilines = digilines,
	description = S("Tractor Beam Generator"),
	tiles = {"traverse_cannon.png^[hsl:30","traverse_cannon.png^[hsl:30","traverse_cannon.png^[hsl:30","traverse_cannon.png^[hsl:30","traverse_cannon.png^[hsl:30","traverse_cannon.png^[hsl:30^traverse_beam_front.png"},
	groups = {cracky = 2},
	paramtype = "light",
	light_source = 12,
	paramtype2 = "facedir",
	on_place = function(itemstack, placer, pointed_thing)
		local player_look_dir = placer:get_look_dir()
		local rotated_facedir = core.dir_to_facedir(player_look_dir,true)
		return core.item_place(itemstack, placer, pointed_thing,rotated_facedir)
	end,
	on_timer = function (pos)
		local dir = core.facedir_to_dir(core.get_node(pos).param2)
		--vector like {y = 0, x = -1, z = 0}
		local meta = core.get_meta(pos)
		local pull = meta:get_int("pull") == 1
		local reach = meta:get_int("reach")
		
		local forcepos = pos
		if pull then forcepos = vector.add(pos, vector.multiply(dir,-1)) end
		
		
		local reach_search = (reach+0.5)*-1
		local reach_move = (reach+1)*-1
		local objects = core.get_objects_in_area(
			{x = forcepos.x + (dir.x<0 and dir.x*reach_search or -0.5), y = forcepos.y + (dir.y<0 and dir.y*reach_search or -0.5)  , z = forcepos.z + (dir.z<0 and dir.z*reach_search or -0.5)},
			{x = forcepos.x + (dir.x>0 and dir.x*reach_search or  0.5), y = forcepos.y + (dir.y>0 and dir.y*reach_search or  0.499), z = forcepos.z + (dir.z>0 and dir.z*reach_search or  0.5)}
		)
		for _, o in pairs(objects) do
			o = remove_mover_if_closer(o,pos)
			
			if o and is_movable(o) then
				local centered = o:get_pos()
				for i=1,3 do
					if dir[i] == 0 then centered[i] = pos[i] end
				end
				
				local staticdata = core.write_json({
					mode = "beam",
					origin = pos,
				})
				local mover = core.add_entity(centered, "traverse:mover", staticdata)
				
				if o:is_player() then
					mover:get_luaentity()._audio = core.sound_play({name = "traverse_beam"},
					{
						to_player = o:get_player_name(),
						loop = true,
					})
				end
				
				o:set_pos(centered)
				o:set_attach(mover)
				local vel = vector.multiply(dir,-5*(pull and -1 or 1))
				--since the default interval limit is 0.2s, 5m/s is the fastest it can go without mobs passing through other beams unnoticed
				mover:set_velocity(vel)
				
				local e = mover:get_luaentity()
				
				e._area = {{x = pos.x + (dir.x<0 and dir.x*reach_move or -0.5), y = pos.y + (dir.y<0 and dir.y*reach_move or -0.5)  , z = pos.z + (dir.z<0 and dir.z*reach_move or -0.5)},
						   {x = pos.x + (dir.x>0 and dir.x*reach_move or  0.5), y = pos.y + (dir.y>0 and dir.y*reach_move or  0.499), z = pos.z + (dir.z>0 and dir.z*reach_move or  0.5)}}
				local area = e._area
				for _,v in ipairs(xyz) do --fix negative areas
					if area[1][v] > area[2][v] then
						local temp = area[1][v]
						area[1][v] = area[2][v]
						area[2][v] = temp
					end
				end
			end
		end
		return true
	end,
	on_construct = function(pos)
		local meta = core.get_meta(pos)
		meta:set_int("reach",10)
		core.get_node_timer(pos):start(0.1)
	end,
	on_rightclick = configure_node,
})


core.register_node("traverse:pipe", {
	description = S("Pipe"),
	tiles = {"traverse_pipe_side.png^[transform2","traverse_pipe_side.png","traverse_pipe_side.png^[transform1","traverse_pipe_side.png^[transform3","traverse_pipe_back.png","traverse_pipe_front.png"},
	drawtype = "nodebox",
	groups = {cracky = 2},
	paramtype2 = "facedir",
	on_place = function(itemstack, placer, pointed_thing)
		local player_look_dir = placer:get_look_dir()
		local rotated_facedir = core.dir_to_facedir(player_look_dir,true)
		return core.item_place(itemstack, placer, pointed_thing,rotated_facedir)
	end,
	selection_box = {
		type = "fixed",
		fixed = {-0.5,-0.5,-0.5,0.5,0.5,0.5},
	},
	node_box = {
		type = "fixed",
		fixed = {
			{-0.5, -0.5, -5/16,  -5/16,  5/16, -0.5,},
			{5/16, -5/16, -5/16,  0.5, 0.5, -0.5},
			{-5/16, -0.5, -5/16,  0.5, -5/16, -0.5, },
			{-0.5,  5/16, -5/16, 5/16, 0.5, -0.5},
			
			{-7/16, -7/16, -5/16,  -5/16,  5/16, 8/16,},
			{5/16, -5/16, -5/16,  7/16, 7/16, 8/16},
			{-5/16, -7/16, -5/16,  7/16, -5/16, 8/16, },
			{-7/16,  5/16, -5/16, 5/16, 7/16, 8/16},
			
			{-5/16,  -5/16, 0.4, 5/16, 5/16, 0.5},
		},
	},
	collision_box = {
		type = "fixed",
		fixed = {
			{-0.5, -0.5, -0.5,  -7/16,  7/16, 0.5,},
			{7/16, -7/16, -0.5,  0.5, 0.5, 0.5},
			{-7/16, -0.5, -0.5,  0.5, -7/16, 0.5, },
			{-0.5,  7/16, -0.5, 7/16, 0.5, 0.5},
			
			{-7/16,  -7/16, 0.4, 7/16, 7/16, 0.5},
		},
	},
	on_construct = function(pos)
		local function buildable_to(n)
			return core.registered_nodes[n.name].buildable_to
		end
		core.get_node_timer(pos):start(0.1)
		
		local dir = core.facedir_to_dir(core.get_node(pos).param2)
		local n
		local v
		local air = 0
		for i=1,100 do
			v = vector.add(pos,vector.multiply(dir,i))
			n = core.get_node(v)
			if n.name == "traverse:pipe" then
				break
			end
			if buildable_to(n) then
				air = air + 1
				if air == 3 then
					--clear nodes before pipe if buildable_to
					local before_pipe = vector.add(pos,vector.multiply(dir,-1))
					if buildable_to(core.get_node(before_pipe)) then core.remove_node(before_pipe) end
					before_pipe = vector.add(pos,vector.multiply(dir,-2))
					if buildable_to(core.get_node(before_pipe)) then core.remove_node(before_pipe) end
				
					--place second pipe part
					core.set_node(
						vector.add(pos,vector.multiply(dir,i-2)),
						{name="traverse:pipe",param2=core.dir_to_facedir(vector.multiply(dir, -1),true)}
					)
					--~ --clear buildable_to nodes after pipe
					core.remove_node(vector.add(pos,vector.multiply(dir,i-1)))
					core.remove_node(vector.add(pos,vector.multiply(dir,i)))
					break
				end
			else
				air = 0
			end
		end
	end,
	after_destruct = function(pos,oldnode)
		local dir = core.facedir_to_dir(oldnode.param2)
		local n
		local v
		for i=1,98 do
			v = vector.add(pos,vector.multiply(dir,i))
			n = core.get_node(v)
			if n.name == "traverse:pipe" then
				core.remove_node(v)
				break
			end
		end
	end,
	on_timer = function (pos)
		local dir = core.facedir_to_dir(core.get_node(pos).param2)
		--vector like {y = 0, x = -1, z = 0}
		
		local start = vector.add(pos,vector.multiply(dir,-1))
		local objects = core.get_objects_in_area(
			vector.add(corner_one,start),
			vector.add(corner_two,start)
		)
		for _, o in pairs(objects) do
			o = remove_mover_if_closer(o,pos)
			if o and is_movable(o) then
				
				local n
				local v
				local air = 0
				for i=1,98 do
					v = vector.add(pos,vector.multiply(dir,i))
					n = core.get_node(v)
					if n.name == "traverse:pipe" then
						if core.get_node(vector.add(v,vector.multiply(dir,1))).name == "air" and core.get_node(vector.add(v,vector.multiply(dir,2))).name == "air" then
							if dir.y == 1 then o:add_velocity({x=4*math.sign(math.random()-0.5),y=10,z=4*math.sign(math.random()-0.5)}) end --prevents player from falling down again when traversing up a pipe
							o:set_pos(vector.add(v,vector.multiply(dir,2)))
							if o:is_player() then
								core.sound_play({name = "traverse_pipe",gain=0.5},
								{
									to_player = o:get_player_name(),
								},true)
							end
						end
						break
					else
						air = 0
					end
				end
			end
		end
		return true
	end,
})

local function rotate_vector_towards_player(vector, player_pos, target_pos)
	local direction = vector.normalize(vector.subtract(target_pos, player_pos))
	local magnitude = vector.length(vector)
	local unit_vector = vector.normalize(vector)
	local rotated_vector = vector.multiply(direction, magnitude)
	return vector.multiply(rotated_vector,-1)
end


core.register_node("traverse:cannon", {
	digilines = digilines,
	description = S("Cannon"),
	tiles = {"traverse_cannon.png"},
	paramtype="light",
	walkable = false,
	groups = {cracky = 2},
	after_place_node = function(pos, placer, itemstack, pointed_thing)
		core.add_entity(pos, "traverse:cannon")
		
		local v = rotate_vector_towards_player(vector.new(20,0,0), placer:get_pos()+vector.new(0,placer:get_properties().eye_height,0), pos)
		local meta = core.get_meta(pos)
		adjust_cannon(pos,meta,math.round(v.x),math.round(v.y),math.round(v.z))
		
		local player_look_dir = placer:get_look_dir()
		local rotated_facedir = core.dir_to_facedir(player_look_dir,true)
		return itemstack
	end,

	on_timer = function (pos)
		local block_pos=pos
		local objects = core.get_objects_in_area(
			{x = block_pos.x - 1, y = block_pos.y - 1, z = block_pos.z - 1},
			{x = block_pos.x + 1, y = block_pos.y + 1, z = block_pos.z + 1}
		)
		for _, o in pairs(objects) do
			o = remove_mover_if_closer(o,pos)
			
			if o and is_movable(o) then
				
				local staticdata = core.write_json({
					mode = "cannon",
					origin = pos,
				})
				local mover = core.add_entity(pos, "traverse:mover", staticdata)
				mover:set_pos(pos)
				o:set_pos(pos)
				o:set_attach(mover)
				
				local meta = core.get_meta(pos)
				local x = meta:get_float("vel_x")
				local y = meta:get_float("vel_y")
				local z = meta:get_float("vel_z")
				
				mover:add_velocity(vector.new(x,y,z))

				
				core.sound_play({name = "traverse_cannon"},
				{
					pos = pos,
				})
			end
		end
		return true
	end,
	on_construct = function(pos)
		core.get_node_timer(pos):start(0.1)
	end,
	on_destruct = function(pos)
		--prevents pipeless cannons when pushed by piston
		core.after(0.1,function(pos)
			local o = get_cannon_entity(pos)
			if o then o:get_luaentity():on_activate() end
		end,pos)
	end,
	on_rightclick = configure_node,
})

core.register_entity("traverse:cannon", {
	initial_properties = {
		physical = false,
		visual = "mesh",
		mesh = "cannon.glb",
		visual_size = vector.new(5,5,5),
		textures = {"traverse_cannon_front.png","traverse_cannon_back.png","traverse_cannon_side.png"},
		collisionbox = {0, 0, 0, 0, 0, 0},
		selectionbox = {0, 0, 0, 0, 0, 0},
	},
	
	on_activate = function(self, staticdata, dtime_s)
		if core.get_node(self.object:get_pos()).name ~= "traverse:cannon" then
			self.object:remove()
		end
	end,
})