-- Copyright (C) 2025  snoutie
-- Authors: snoutie (copyright@achtarmig.org)
-- This program is free software: you can redistribute it and/or modify
-- it under the terms of the GNU Affero General Public License as published
-- by the Free Software Foundation, either version 3 of the License, or
-- (at your option) any later version.

-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU Affero General Public License for more details.

-- You should have received a copy of the GNU Affero General Public License
-- along with this program.  If not, see <https://www.gnu.org/licenses/>.

--Stores liquid_id and the corresponding liquid node names
liquid_physics.registered_liquids = {}

--Stores all liquid node names and their corresponding liquid_id
local liquid_ids = {}

-- Returns liquid_id when node is a liquid, 0 when node is air
-- and nil when node is anything else
-- Note: 0 is not a valid ID
function liquid_physics.get_liquid_id(node_name)
    if node_name == "air" then
        return 0
    end
    return liquid_ids[node_name]
end

--Returns name of liquid node. Note: if liquid_level == 0 then "air" will be returned
function liquid_physics.get_liquid_node_name(liquid_id, liquid_level)
    if liquid_level == 0 then
        return "air"
    end

    return liquid_physics.registered_liquids[liquid_id][liquid_level + 1]
end

--Returns the liquid level. If the node is air, 0 will be returned.
-- -1 will be returned for any other node
function liquid_physics.get_liquid_level(liquid_id, node_name)
    if node_name == "air" then
        return 0
    end
    if liquid_physics.registered_liquids[liquid_id] == nil then
        error(liquid_id)
    end
    for i = 1, 9 do
        if node_name == liquid_physics.registered_liquids[liquid_id][i] then
            return i - 1
        end
    end

    return -1
end

--@param namespace string Name of mod eg. default
--@param name string Name of former liquid source
-- Executing this function will override the liquid source specified.
-- New liquid sources will be "liquid_physics:namespace_name_level_1" through 7
function liquid_physics.register_liquid(namespace, source_name, flowing_name)
    --Get next id
    local id = table.getn(liquid_physics.registered_liquids) + 1

    --Get source definition
    local source_liquid_name = namespace .. ":" .. source_name
    local source_liquid_def = core.registered_nodes[source_liquid_name]

    if source_liquid_def.liquidtype ~= "source" then
        error("Liquid Physics: " ..
            source_liquid_name .. " is of type " .. source_liquid_def.drawtype .. " and not of type source")
    end

    --Get flowing definiton
    local flowing_liquid_name = namespace .. ":" .. flowing_name
    local flowing_liquid_def = core.registered_nodes[flowing_liquid_name]

    if flowing_liquid_def.liquidtype ~= "flowing" then
        error("Liquid Physics: " ..
            source_liquid_name .. " is of type " .. source_liquid_def.drawtype .. " and not of type source")
    end

    --Overwrite source
    source_liquid_def.liquid_range = 0
    source_liquid_def.liquid_alternative_flowing = source_liquid_name
    source_liquid_def.liquid_alternative_source = source_liquid_name

    core.register_node(":" .. source_liquid_name, source_liquid_def)
    liquid_ids[source_liquid_name] = id

    local liquids = {}
    table.insert(liquids, "air")

    for i = 1, 7 do
        local node_name = "liquid_physics:" .. namespace .. "_" .. source_name .. "_level_" .. i

        local level_def = {
            description = source_liquid_def.description .. " Level " .. i,
            tiles = source_liquid_def.tiles,
            use_texture_alpha = source_liquid_def.use_texture_alpha,
            paramtype = source_liquid_def.paramtype,
            leveled = 8 * i,
            walkable = source_liquid_def.walkable,
            pointable = source_liquid_def.pointable,
            diggable = source_liquid_def.diggable,
            buildable_to = source_liquid_def.buildable_to,
            is_ground_content = source_liquid_def.is_ground_content,
            drawtype = "nodebox",
            drop = source_liquid_def.drop,
            drowning = source_liquid_def.drowning,
            liquidtype = "water_source",
            liquid_alternative_flowing = node_name,
            liquid_alternative_source = node_name,
            liquid_range = 0,
            liquid_viscosity = source_liquid_def.liquid_viscosity,
            waving = source_liquid_def.waving,
            color = source_liquid_def.color,
            node_box = {
                type = "leveled",
                fixed = {
                    { -0.5, -0.5, -0.5, 0.5, -0.375 + (0.125 * (i - 1)), 0.5 },
                },
            },
            collision_box = {
                type = "fixed",
                fixed = { { -0.5, -0.5, -0.5, 0.5, -0.375 + (0.125 * (i - 1)), 0.5 }, },
            },
            post_effect_color = source_liquid_def.post_effect_color,
            groups = source_liquid_def.groups,
            sounds = source_liquid_def.sounds,
        }

        core.register_node(node_name, level_def)
        liquid_ids[node_name] = id

        table.insert(liquids, node_name)
    end

    table.insert(liquids, source_liquid_name)

    --Finally, replace flowing, such that it behaves like a source
    flowing_liquid_def.liquidtype = "source"
    flowing_liquid_def.liquid_range = 0
    flowing_liquid_def.liquid_alternative_flowing = source_liquid_name
    flowing_liquid_def.liquid_alternative_source = source_liquid_name

    core.register_node(":" .. flowing_liquid_name, flowing_liquid_def)
    liquid_ids[flowing_liquid_name] = id

    liquid_physics.registered_liquids[id] = liquids
end
