-- Setup
local modname = minetest.get_current_modname()
local modpath = minetest.get_modpath(modname)
local S = minetest.get_translator(modname)
local version = "1.2"

-- CHANGELOG
-- * v1.0: Initial version ported from mcl_rocky_tools w/ netherrack
--     pick.
-- * v1.1: Change to license file (code file still says v1.0).
-- * v1.2: Fixed crash on sand fall bug where netherrack ondig gets
--     called with nil digger.

-- Help texts
local pickaxe_longdesc = S("Pickaxes made from semi-rare stones like Andesite or Granite have slightly better stats than cobblestone pickaxes, but they will not mine any additional ores (use iron for that).")
local shovel_longdesc = S("Shovels made from semi-rare stones like Andesite or Granite have slightly better stats than cobblestone axes.")
local axe_longdesc = S("Axes made from semi-rare stones like Andesite or Granite have slightly better stats than cobblestone axes.")
local hoe_longdesc = S("Hoes made from semi-rare stones like Andesite or Granite have slightly better stats than cobblestone hoes.")

local hoe_usagehelp = S("Use the hoe on a cultivatable block (by rightclicking it) to turn it into farmland. Dirt, grass blocks and grass paths are cultivatable blocks. Using a hoe on coarse dirt turns it into dirt.")
local shovel_use = S("To turn a grass block into a grass path, hold the shovel in your hand, then use (rightclick) the top or side of a grass block. This only works when there's air above the grass block.")

local wield_scale = mcl_vars.tool_wield_scale

-- Changes from base stone pickaxe:
-- enchantability up from 5 to 8 for all
-- punch_attack_uses up from 66 to 75 for all
-- uses up from 132 to 183 for all

-- Variant-specific changes:
-- Andesite gets range up from 4.0 to 4.5
-- Diorite gets speed up from 4 to 4.5
-- Granite gets punch_attack_uses up to 93 and uses up to 211
-- Deepslate gets speed up from 4 to 5, punch_attack_uses up to 104, and
--   uses up to 237
-- Netherrack get speed up from 4 to 4.5, but uses down to 99. Has 5%
--   chance to set the block below on fire when mining and deals damage
--   like a sword.

-- From:
-- https://stackoverflow.com/questions/2421695/first-character-uppercase-lua
function firstToUpper(str)
    return (str:gsub("^%l", string.upper))
end


-- TODO: This, the right way.
local function create_soil(pos, inv)
	if pos == nil then
		return false
	end
	local node = minetest.get_node(pos)
	local name = node.name
	local above = minetest.get_node({x=pos.x, y=pos.y+1, z=pos.z})
	if minetest.get_item_group(name, "cultivatable") == 2 then
		if above.name == "air" then
			node.name = "mcl_farming:soil"
			minetest.set_node(pos, node)
			minetest.sound_play("default_dig_crumbly", { pos = pos, gain = 0.5 }, true)
			return true
		end
	elseif minetest.get_item_group(name, "cultivatable") == 1 then
		if above.name == "air" then
			node.name = "mcl_core:dirt"
			minetest.set_node(pos, node)
			minetest.sound_play("default_dig_crumbly", { pos = pos, gain = 0.6 }, true)
			return true
		end
	end
	return false
end

local hoe_on_place_function = function(wear_divisor)
	return function(itemstack, user, pointed_thing)
		-- Call on_rightclick if the pointed node defines it
		local node = minetest.get_node(pointed_thing.under)
		if user and not user:get_player_control().sneak then
			if minetest.registered_nodes[node.name] and minetest.registered_nodes[node.name].on_rightclick then
				return minetest.registered_nodes[node.name].on_rightclick(pointed_thing.under, node, user, itemstack) or itemstack
			end
		end

		if minetest.is_protected(pointed_thing.under, user:get_player_name()) then
			minetest.record_protection_violation(pointed_thing.under, user:get_player_name())
			return itemstack
		end

		if create_soil(pointed_thing.under, user:get_inventory()) then
			if not minetest.is_creative_enabled(user:get_player_name()) then
				itemstack:add_wear(65535/wear_divisor)
			end
			return itemstack
		end
	end
end

local make_grass_path = function(itemstack, placer, pointed_thing)
	-- Use pointed node's on_rightclick function first, if present
	local node = minetest.get_node(pointed_thing.under)
	if placer and not placer:get_player_control().sneak then
		if minetest.registered_nodes[node.name] and minetest.registered_nodes[node.name].on_rightclick then
			return minetest.registered_nodes[node.name].on_rightclick(pointed_thing.under, node, placer, itemstack) or itemstack
		end
	end

	-- Only make or remove grass path if tool used on side or top of target node
	if pointed_thing.above.y < pointed_thing.under.y then
		return itemstack
	end

	-- Remove grass paths
	if (minetest.get_item_group(node.name, "path_remove_possible") == 1) and placer:get_player_control().sneak then
		local above = table.copy(pointed_thing.under)
		above.y = above.y + 1
		if minetest.get_node(above).name == "air" then
			if minetest.is_protected(pointed_thing.under, placer:get_player_name()) then
				minetest.record_protection_violation(pointed_thing.under, placer:get_player_name())
				return itemstack
			end

			if not minetest.is_creative_enabled(placer:get_player_name()) then
				-- Add wear (as if digging a shovely node)
				local toolname = itemstack:get_name()
				local wear = mcl_autogroup.get_wear(toolname, "shovely")
				if wear then
					itemstack:add_wear(wear)
					tt.reload_itemstack_description(itemstack) -- update tooltip
				end
			end
			minetest.sound_play({name="default_grass_footstep", gain=1}, {pos = above, max_hear_distance = 16}, true)
			minetest.swap_node(pointed_thing.under, {name="mcl_core:dirt"})
		end
	end

	-- Make grass paths
	if (minetest.get_item_group(node.name, "path_creation_possible") == 1) and not placer:get_player_control().sneak then
		local above = table.copy(pointed_thing.under)
		above.y = above.y + 1
		if minetest.get_node(above).name == "air" then
			if minetest.is_protected(pointed_thing.under, placer:get_player_name()) then
				minetest.record_protection_violation(pointed_thing.under, placer:get_player_name())
				return itemstack
			end

			if not minetest.is_creative_enabled(placer:get_player_name()) then
				-- Add wear (as if digging a shovely node)
				local toolname = itemstack:get_name()
				local wear = mcl_autogroup.get_wear(toolname, "shovely")
				if wear then
					itemstack:add_wear(wear)
					tt.reload_itemstack_description(itemstack) -- update tooltip
				end
			end
			minetest.sound_play({name="default_grass_footstep", gain=1}, {pos = above, max_hear_distance = 16}, true)
			minetest.swap_node(pointed_thing.under, {name="mcl_core:grass_path"})
		end
	end
	return itemstack
end

local function make_stripped_trunk(itemstack, placer, pointed_thing)
	if pointed_thing.type ~= "node" then return end

	local node = minetest.get_node(pointed_thing.under)
	local node_name = minetest.get_node(pointed_thing.under).name

	local noddef = minetest.registered_nodes[node_name]

	if not noddef then
		minetest.log("warning", "Trying to right click with an axe the unregistered node: " .. tostring(node_name))
		return
	end

	if not placer:get_player_control().sneak and noddef.on_rightclick then
		return minetest.item_place(itemstack, placer, pointed_thing)
	end
	if minetest.is_protected(pointed_thing.under, placer:get_player_name()) then
		minetest.record_protection_violation(pointed_thing.under, placer:get_player_name())
		return itemstack
	end

	if noddef._mcl_stripped_variant == nil then
		return itemstack
	else
		minetest.swap_node(pointed_thing.under, {name=noddef._mcl_stripped_variant, param2=node.param2})
		if minetest.get_item_group(node_name, "waxed") ~= 0 then
			awards.unlock(placer:get_player_name(), "mcl:wax_off")
		end
		if not minetest.is_creative_enabled(placer:get_player_name()) then
			-- Add wear (as if digging a axey node)
			local toolname = itemstack:get_name()
			local wear = mcl_autogroup.get_wear(toolname, "axey")
			if wear then
				itemstack:add_wear(wear)
				tt.reload_itemstack_description(itemstack) -- update tooltip
			end
		end
	end
	return itemstack
end


-- Registers the tool + crafting recipe for pickaxe, axe, and shovel for
-- a specific material. Arguments control extra values for a few
-- parameters: range, speed, punches, uses.
local function make_tool_variant(mat, range, speed, punches, uses)
    local full_mat = 'mcl_core:' .. mat;
    if mat == "deepslate" then
        full_mat = 'mcl_deepslate:deepslate_cobbled';
    end
    range = 4.0 + range;
    speed = 4 + speed;
    punches = 72 + punches;
    uses = 174 + uses;

    minetest.register_tool('vl_rocky_tools:pick_' .. mat, {
        description = S(firstToUpper(mat) .. ' Pickaxe'),
        _doc_items_longdesc = pickaxe_longdesc,
        inventory_image = 'vl_rocky_tools_pickaxe_' .. mat .. '.png',
        wield_scale = wield_scale,
        -- Slightly longer-than-default range.
        range=range,
        groups = { tool=1, pickaxe=1, dig_speed_class=3, enchantability=8 },
        tool_capabilities = {
            -- 1/1.2
            full_punch_interval = 0.83333333,
            max_drop_level=3,
            damage_groups = {fleshy=3},
            punch_attack_uses = punches
        },
        sound = { breaks = 'default_tool_breaks' },
        _repair_material = full_mat,
        _mcl_toollike_wield = true,
        _mcl_diggroups = {
            pickaxey = { speed = speed, level = 3, uses = uses }
        },
    });

    minetest.register_tool("vl_rocky_tools:shovel_" .. mat, {
        description = S(firstToUpper(mat) .. " Shovel"),
        _doc_items_longdesc = shovel_longdesc,
        _doc_items_usagehelp = shovel_use,
        inventory_image = 'vl_rocky_tools_shovel_' .. mat .. '.png',
        wield_scale = wield_scale,
        groups = { tool=1, shovel=1, dig_speed_class=3, enchantability=8 },
        tool_capabilities = {
            full_punch_interval = 1,
            max_drop_level=3,
            damage_groups = {fleshy=3},
            punch_attack_uses = punches,
        },
        on_place = make_grass_path,
        sound = { breaks = "default_tool_breaks" },
        _repair_material = full_mat,
        _mcl_toollike_wield = true,
        _mcl_diggroups = {
            shovely = { speed = speed, level = 3, uses = uses }
        },
    });

    minetest.register_tool("vl_rocky_tools:axe_" .. mat, {
        description = S(firstToUpper(mat) .. " Axe"),
        _doc_items_longdesc = axe_longdesc,
        inventory_image = 'vl_rocky_tools_axe_' .. mat .. '.png',
        wield_scale = wield_scale,
        groups = { tool=1, axe=1, dig_speed_class=3, enchantability=8 },
        tool_capabilities = {
            full_punch_interval = 1.25,
            max_drop_level=3,
            damage_groups = {fleshy=9},
            punch_attack_uses = punches,
        },
        on_place = make_stripped_trunk,
        sound = { breaks = "default_tool_breaks" },
        _repair_material = full_mat,
        _mcl_toollike_wield = true,
        _mcl_diggroups = {
            axey = { speed = speed, level = 3, uses = uses }
        },
    });

    minetest.register_tool("vl_rocky_tools:hoe_" .. mat, {
        description = S(firstToUpper(mat) .. " Hoe"),
        _tt_help = (
            S("Turns block into farmland")
         .. "\n"
         .. S("Uses: @1", uses)
        ),
        _doc_items_longdesc = hoe_longdesc,
        _doc_items_usagehelp = hoe_usagehelp,
        inventory_image = 'vl_rocky_tools_hoe_' .. mat .. '.png',
        wield_scale = wield_scale,
        on_place = hoe_on_place_function(uses),
        groups = { tool=1, hoe=1, enchantability=8 },
        tool_capabilities = {
            full_punch_interval = 0.5,
            damage_groups = { fleshy = 1, },
            punch_attack_uses = punches,
        },
        _repair_material = full_mat,
        _mcl_toollike_wield = true,
        _mcl_diggroups = {
            hoey = { speed = speed, level = 3, uses = uses }
        },
    });

    -- Crafting
    minetest.register_craft({
        output = 'vl_rocky_tools:pick_' .. mat,
        recipe = {
            {full_mat, full_mat, full_mat},
            {"", "mcl_core:stick", ""},
            {"", "mcl_core:stick", ""},
        }
    });

    minetest.register_craft({
        output = 'vl_rocky_tools:shovel_' .. mat,
        recipe = {
            {"", full_mat, ""},
            {"", "mcl_core:stick", ""},
            {"", "mcl_core:stick", ""},
        }
    });

    minetest.register_craft({
        output = 'vl_rocky_tools:axe_' .. mat,
        recipe = {
            {full_mat, full_mat, ""},
            {full_mat, "mcl_core:stick", ""},
            {"", "mcl_core:stick", ""},
        }
    });

    minetest.register_craft({
        output = 'vl_rocky_tools:hoe_' .. mat,
        recipe = {
            {full_mat, full_mat, ""},
            {"", "mcl_core:stick", ""},
            {"", "mcl_core:stick", ""},
        }
    });
end

-- Actually create the variants:
make_tool_variant("andesite", 0.5, 0, 0, 0);
make_tool_variant("diorite", 0.0, 0.5, 0, 0);
make_tool_variant("granite", 0.0, 0, 18, 28);
-- Unregister existing deepslate tool crafting recipes we're replacing
-- Note hammers and spears from vl_deepslate_tools are left alone
minetest.clear_craft({output = 'vl_deepslate_tools:pick_deepslate'});
minetest.clear_craft({output = 'vl_deepslate_tools:shovel_deepslate'});
minetest.clear_craft({output = 'vl_deepslate_tools:axe_deepslate'});
minetest.clear_craft({output = 'vl_deepslate_tools:hoe_deepslate'});
make_tool_variant("deepslate", 0.0, 0.8, 29, 54);

-- Nethrerack pickaxe has a 5% chance to set the square on fire after
local n = 'mcl_core:netherrack';
minetest.register_tool('vl_rocky_tools:pick_netherrack', {
    description = S('Netherrack Pickaxe'),
    _doc_items_longdesc = pickaxe_longdesc,
    inventory_image = 'vl_rocky_tools_pickaxe_netherrack.png',
    wield_scale = wield_scale,
    -- Slightly longer-than-default range.
    range=4.5,
    groups = { tool=1, pickaxe=1, dig_speed_class=2, enchantability=3 },
    tool_capabilities = {
        -- 1/1.2
        full_punch_interval = 0.83333333,
        max_drop_level=3,
        damage_groups = {fleshy=5},  -- as much as a sword
        punch_attack_uses = 70  -- not too many
    },
    sound = { breaks = 'default_tool_breaks' },
    _repair_material = n,
    _mcl_toollike_wield = true,
    _mcl_diggroups = {
        -- faster
        pickaxey = { speed = 4.5, level = 3, uses = 99 },
        -- To slow down mining for by-hand stuff (not necessary):
        -- handy = { speed = 1, level = 1, uses = 1000 }
    },
});

-- Register global after_dig callback because after_use callback doesn't
-- get enough info to figure out the coords of the node destroyed...
local function burn_after_digging_with_netherrack(pos, oldnode, digger)
    if digger == nil then
        return
    end
    local stack = digger:get_wielded_item();
    if stack:get_name() == "vl_rocky_tools:pick_netherrack" then
        -- 8% chance to light a fire, but always re-lights if we just
        -- destroyed a fire
        if math.random() < 0.08 or oldnode.name == "mcl_fire:fire" then
            core.add_node(pos,  { name = "mcl_fire:fire", param2=254 });
            minetest.sound_play(
                "fire_flint_and_steel",
                {pos = pos, gain = 0.5, max_hear_distance = 8},
                true
            )
        end
    end
end

minetest.register_on_dignode(burn_after_digging_with_netherrack);

minetest.register_craft({
    output = 'vl_rocky_tools:pick_netherrack',
    recipe = {
        {n, n, n},
        {"", "mcl_core:stick", ""},
        {"", "mcl_core:stick", ""},
    }
});

-- TODO: Fire chance
-- TODO: shovel/axe/hoe? LOL

-- Ideas:
-- Sandstone pickaxe is just bad? Or also serves as a shovel?
--   Can be crafted without a crafting table?
-- Obsidan? Higher tier?
-- End Stone Pickaxe? What would it do?
-- Does MC2 have these?
--   Dripstone?
--   Basalt?
--   Calcite?
