
local minetest = minetest

buildings = {}

buildings.sky_above = function(pos)
	-- Get light at node above during midday (which should be equal to sun light)
	return minetest.get_node_light({x=pos.x, y=pos.y+0.5, z=pos.z}, 0.5) == 15
end

local altar1 = {
	{{"group:red"},
	{
		-- center has 3 red
		{x=0, y=0, z=0},
		{x=0, y=1, z=0},
		{x=0, y=-1, z=0},

		-- each of 8 small pillars has 1 red
		{x=2, y=-1, z=1},
		{x=2, y=-1, z=-1},
		{x=-2, y=-1, z=1},
		{x=-2, y=-1, z=-1},
		{x=1, y=-1, z=2},
		{x=1, y=-1, z=-2},
		{x=-1, y=-1, z=2},
		{x=-1, y=-1, z=-2}
	}},
	{{"group:yellow"},
	{
		-- all yellow is on ground
		-- around center
		{x=1, y=-2, z=0},
		{x=-1, y=-2, z=0},
		{x=1, y=-2, z=1},
		{x=-1, y=-2, z=1},
		{x=1, y=-2, z=-1},
		{x=-1, y=-2, z=1},
		{x=0, y=-2, z=1},
		{x=0, y=-2, z=-1},

		-- around the whole thing
		{x=4, y=-2, z=4},
		{x=4, y=-2, z=3},
		{x=4, y=-2, z=2},
		{x=4, y=-2, z=1},
		{x=4, y=-2, z=0},
		{x=4, y=-2, z=-1},
		{x=4, y=-2, z=-2},
		{x=4, y=-2, z=-3},

		{x=4, y=-2, z=-4},
		{x=3, y=-2, z=-4},
		{x=2, y=-2, z=-4},
		{x=1, y=-2, z=-4},
		{x=0, y=-2, z=-4},
		{x=-1, y=-2, z=-4},
		{x=-2, y=-2, z=-4},
		{x=-3, y=-2, z=-4},

		{x=-4, y=-2, z=-4},
		{x=-4, y=-2, z=-3},
		{x=-4, y=-2, z=-2},
		{x=-4, y=-2, z=-1},
		{x=-4, y=-2, z=0},
		{x=-4, y=-2, z=1},
		{x=-4, y=-2, z=2},
		{x=-4, y=-2, z=3},

		{x=-4, y=-2, z=4},
		{x=-3, y=-2, z=4},
		{x=-2, y=-2, z=4},
		{x=-1, y=-2, z=4},
		{x=0, y=-2, z=4},
		{x=1, y=-2, z=4},
		{x=2, y=-2, z=4},
		{x=3, y=-2, z=4}
	}},
	{{"group:green"},
	{
		-- green on top of all pillars
		{x=0, y=2, z=0},
		{x=2, y=0, z=1},
		{x=2, y=0, z=-1},
		{x=-2, y=0, z=1},
		{x=-2, y=0, z=-1},
		{x=1, y=0, z=2},
		{x=1, y=0, z=-2},
		{x=-1, y=0, z=2},
		{x=-1, y=0, z=-2},

		-- green around the thing (but inside the yellow)
		{x=3, y=-2, z=3},
		{x=3, y=-2, z=2},
		{x=3, y=-2, z=1},
		{x=3, y=-2, z=0},
		{x=3, y=-2, z=-1},
		{x=3, y=-2, z=-2},

		{x=3, y=-2, z=-3},
		{x=2, y=-2, z=-3},
		{x=1, y=-2, z=-3},
		{x=0, y=-2, z=-3},
		{x=-1, y=-2, z=-3},
		{x=-2, y=-2, z=-3},

		{x=-3, y=-2, z=-3},
		{x=-3, y=-2, z=-2},
		{x=-3, y=-2, z=-1},
		{x=-3, y=-2, z=0},
		{x=-3, y=-2, z=1},
		{x=-3, y=-2, z=2},

		{x=-3, y=-2, z=3},
		{x=-2, y=-2, z=3},
		{x=-1, y=-2, z=3},
		{x=0, y=-2, z=3},
		{x=1, y=-2, z=3},
		{x=2, y=-2, z=3}
	}},
	{{"group:black"},
	{
		-- entrances
		{x=2, y=-2, z=0},
		{x=-2, y=-2, z=0},
		{x=0, y=-2, z=2},
		{x=0, y=-2, z=-2},

		-- exterior corners
		{x=2, y=-2, z=2},
		{x=-2, y=-2, z=2},
		{x=2, y=-2, z=-2},
		{x=-2, y=-2, z=-2}
	}}
}

-- # = Black
-- o = Yellow

-- From side (any)
--     G
--     R
--   GGRGG
--   RRRRR
-- ooooooooo

-- From top
-- ooooooooo
-- oGGGGGGGo
-- oG#G#G#Go
-- oGGoooGGo
-- oG#oGo#Go
-- oGGoooGGo
-- oG#G#G#Go
-- ooooooooo

-- Heightmap
-- 000000000
-- 000000000
-- 000202000
-- 002000200
-- 000040000
-- 002000200
-- 000202000
-- 000000000
-- 000000000

local altar2 = {
	{{"group:red"},
	{
		-- all red is on ground
		-- red is crisscrossing across the floor
		{x=1, y=-2, z=1},
		{x=1, y=-2, z=-1},
		{x=-1, y=-2, z=1},
		{x=-1, y=-2, z=-1},

		{x=2, y=-2, z=0},
		{x=-2, y=-2, z=0},
		{x=0, y=-2, z=2},
		{x=0, y=-2, z=-2},

		{x=2, y=-2, z=2},
		{x=-2, y=-2, z=2},
		{x=2, y=-2, z=-2},
		{x=-2, y=-2, z=-2},

		{x=3, y=-2, z=-1},
		{x=3, y=-2, z=1},
		{x=-3, y=-2, z=-1},
		{x=-3, y=-2, z=1},

		{x=-1, y=-2, z=3},
		{x=1, y=-2, z=3},
		{x=-1, y=-2, z=-3},
		{x=1, y=-2, z=-3}
	}},
	{{"group:yellow"},
	{
		-- all yellow is on ground
		-- near center
		{x=1, y=-2, z=0},
		{x=-1, y=-2, z=0},
		{x=0, y=-2, z=1},
		{x=0, y=-2, z=-1},

		-- around the whole thing
		{x=4, y=-2, z=4},
		{x=4, y=-2, z=3},
		{x=4, y=-2, z=2},
		{x=4, y=-2, z=1},
		{x=4, y=-2, z=0},
		{x=4, y=-2, z=-1},
		{x=4, y=-2, z=-2},
		{x=4, y=-2, z=-3},

		{x=4, y=-2, z=-4},
		{x=3, y=-2, z=-4},
		{x=2, y=-2, z=-4},
		{x=1, y=-2, z=-4},
		{x=0, y=-2, z=-4},
		{x=-1, y=-2, z=-4},
		{x=-2, y=-2, z=-4},
		{x=-3, y=-2, z=-4},

		{x=-4, y=-2, z=-4},
		{x=-4, y=-2, z=-3},
		{x=-4, y=-2, z=-2},
		{x=-4, y=-2, z=-1},
		{x=-4, y=-2, z=0},
		{x=-4, y=-2, z=1},
		{x=-4, y=-2, z=2},
		{x=-4, y=-2, z=3},

		{x=-4, y=-2, z=4},
		{x=-3, y=-2, z=4},
		{x=-2, y=-2, z=4},
		{x=-1, y=-2, z=4},
		{x=0, y=-2, z=4},
		{x=1, y=-2, z=4},
		{x=2, y=-2, z=4},
		{x=3, y=-2, z=4}
	}},
	{{"group:green"},
	{
		-- green on top of all pillars
		{x=0, y=2, z=0},

		{x=3, y=0, z=0},
		{x=-3, y=0, z=0},
		{x=0, y=0, z=3},
		{x=0, y=0, z=-3},
		{x=3, y=0, z=3},
		{x=3, y=0, z=-3},
		{x=-3, y=0, z=3},
		{x=-3, y=0, z=-3}
	}},
	{{"group:blue"},
	{
		-- pillars are built from blue
		-- center has 3 blue
		{x=0, y=0, z=0},
		{x=0, y=1, z=0},
		{x=0, y=-1, z=0},

		-- each of 8 small pillars has 1 blue
		{x=3, y=-1, z=0},
		{x=-3, y=-1, z=0},
		{x=0, y=-1, z=3},
		{x=0, y=-1, z=-3},
		{x=3, y=-1, z=3},
		{x=3, y=-1, z=-3},
		{x=-3, y=-1, z=3},
		{x=-3, y=-1, z=-3}
	}},
	{{"group:black"},
	{
		-- all black is on ground
		-- criss-crossing like red
		{x=2, y=-2, z=-1},
		{x=2, y=-2, z=1},
		{x=-2, y=-2, z=-1},
		{x=-2, y=-2, z=1},

		{x=-1, y=-2, z=2},
		{x=1, y=-2, z=2},
		{x=-1, y=-2, z=-2},
		{x=1, y=-2, z=-2},


		{x=3, y=-2, z=-2},
		{x=3, y=-2, z=2},
		{x=-3, y=-2, z=-2},
		{x=-3, y=-2, z=2},

		{x=-2, y=-2, z=3},
		{x=2, y=-2, z=3},
		{x=-2, y=-2, z=-3},
		{x=2, y=-2, z=-3}
	}}
}


local ion_cannon = {
	{{"group:yellow"},
	{
		-- all yellow is on ground
		-- around the whole thing, except entraces
		{x=3, y=-2, z=3},
		{x=3, y=-2, z=2},
		{x=3, y=-2, z=1},
		--{x=3, y=-2, z=0},
		{x=3, y=-2, z=-1},
		{x=3, y=-2, z=-2},

		{x=3, y=-2, z=-3},
		{x=2, y=-2, z=-3},
		{x=1, y=-2, z=-3},
		--{x=0, y=-2, z=-3},
		{x=-1, y=-2, z=-3},
		{x=-2, y=-2, z=-3},

		{x=-3, y=-2, z=-3},
		{x=-3, y=-2, z=-2},
		{x=-3, y=-2, z=-1},
		--{x=-3, y=-2, z=0},
		{x=-3, y=-2, z=1},
		{x=-3, y=-2, z=2},

		{x=-3, y=-2, z=3},
		{x=-2, y=-2, z=3},
		{x=-1, y=-2, z=3},
		--{x=0, y=-2, z=3},
		{x=1, y=-2, z=3},
		{x=2, y=-2, z=3}
	}},
	{{"group:purple"},
	{
		-- pillar is built from purple
		-- center has 3 purple
		{x=0, y=0, z=0},
		{x=0, y=1, z=0},
		{x=0, y=-1, z=0}
	}},
	{{"group:black"},
	{
		-- all black is on ground near center
		{x=1, y=-2, z=-1},
		{x=1, y=-2, z=1},
		{x=-1, y=-2, z=-1},
		{x=-1, y=-2, z=1}
	}},
	{{"group:blue"},
	{
		-- all blue is on ground
		-- next to center
		{x=1, y=-2, z=0},
		{x=-1, y=-2, z=0},
		{x=0, y=-2, z=1},
		{x=0, y=-2, z=-1},

		-- inner ring
		{x=2, y=-2, z=2},
		{x=2, y=-2, z=1},
		{x=2, y=-2, z=0},
		{x=2, y=-2, z=-1},

		{x=2, y=-2, z=-2},
		{x=1, y=-2, z=-2},
		{x=0, y=-2, z=-2},
		{x=-1, y=-2, z=-2},

		{x=-2, y=-2, z=-2},
		{x=-2, y=-2, z=-1},
		{x=-2, y=-2, z=0},
		{x=-2, y=-2, z=1},

		{x=-2, y=-2, z=2},
		{x=-1, y=-2, z=2},
		{x=0, y=-2, z=2},
		{x=1, y=-2, z=2},

		-- entrances
		{x=3, y=-2, z=0},
		{x=0, y=-2, z=-3},
		{x=-3, y=-2, z=0},
		{x=0, y=-2, z=3}
	}},
	{{"group:red"},
	{
		-- single red in the center to activate everything
		{x=0, y=-2, z=0}
	}}
}


local altar3 = {
	{{"group:yellow"},
	{
		-- all yellow is on ground
		-- stretching out from center
		{x=1, y=-2, z=0},
		{x=2, y=-2, z=0},
		{x=3, y=-2, z=0},
		{x=4, y=-2, z=0},
		{x=5, y=-2, z=0},
		{x=6, y=-2, z=0},

		{x=-1, y=-2, z=0},
		{x=-2, y=-2, z=0},
		{x=-3, y=-2, z=0},
		{x=-4, y=-2, z=0},
		{x=-5, y=-2, z=0},
		{x=-6, y=-2, z=0},

		{x=0, y=-2, z=1},
		{x=0, y=-2, z=2},
		{x=0, y=-2, z=3},
		{x=0, y=-2, z=4},
		{x=0, y=-2, z=5},
		{x=0, y=-2, z=6},

		{x=0, y=-2, z=-1},
		{x=0, y=-2, z=-2},
		{x=0, y=-2, z=-3},
		{x=0, y=-2, z=-4},
		{x=0, y=-2, z=-5},
		{x=0, y=-2, z=-6},

		-- in inner corner of walls
		{x=3, y=-2, z=2},
		{x=3, y=-2, z=-2},
		{x=-3, y=-2, z=2},
		{x=-3, y=-2, z=-2},
		{x=2, y=-2, z=3},
		{x=-2, y=-2, z=3},
		{x=2, y=-2, z=-3},
		{x=-2, y=-2, z=-3},

		-- around the whole thing
		{x=7, y=-2, z=7},
		{x=7, y=-2, z=6},
		{x=7, y=-2, z=5},
		{x=7, y=-2, z=4},
		{x=7, y=-2, z=3},
		{x=7, y=-2, z=2},
		{x=7, y=-2, z=1},
		{x=7, y=-2, z=0},
		{x=7, y=-2, z=-1},
		{x=7, y=-2, z=-2},
		{x=7, y=-2, z=-3},
		{x=7, y=-2, z=-4},
		{x=7, y=-2, z=-5},
		{x=7, y=-2, z=-6},

		{x=7, y=-2, z=-7},
		{x=6, y=-2, z=-7},
		{x=5, y=-2, z=-7},
		{x=4, y=-2, z=-7},
		{x=3, y=-2, z=-7},
		{x=2, y=-2, z=-7},
		{x=1, y=-2, z=-7},
		{x=0, y=-2, z=-7},
		{x=-1, y=-2, z=-7},
		{x=-2, y=-2, z=-7},
		{x=-3, y=-2, z=-7},
		{x=-4, y=-2, z=-7},
		{x=-5, y=-2, z=-7},
		{x=-6, y=-2, z=-7},

		{x=-7, y=-2, z=-7},
		{x=-7, y=-2, z=-6},
		{x=-7, y=-2, z=-5},
		{x=-7, y=-2, z=-4},
		{x=-7, y=-2, z=-3},
		{x=-7, y=-2, z=-2},
		{x=-7, y=-2, z=-1},
		{x=-7, y=-2, z=0},
		{x=-7, y=-2, z=1},
		{x=-7, y=-2, z=2},
		{x=-7, y=-2, z=3},
		{x=-7, y=-2, z=4},
		{x=-7, y=-2, z=5},
		{x=-7, y=-2, z=6},

		{x=-7, y=-2, z=7},
		{x=-6, y=-2, z=7},
		{x=-5, y=-2, z=7},
		{x=-4, y=-2, z=7},
		{x=-3, y=-2, z=7},
		{x=-2, y=-2, z=7},
		{x=-1, y=-2, z=7},
		{x=0, y=-2, z=7},
		{x=1, y=-2, z=7},
		{x=2, y=-2, z=7},
		{x=3, y=-2, z=7},
		{x=4, y=-2, z=7},
		{x=5, y=-2, z=7},
		{x=6, y=-2, z=7}
	}},
	{{"group:green"},
	{
		-- green on top of the central pillar
		{x=0, y=3, z=0}
	}},
	{{"group:blue"},
	{
		-- blue in central pillar and on ground
		-- center has 2 blue
		{x=0, y=0, z=0},
		{x=0, y=-1, z=0},

		-- the rest is on ground
		-- a path on the ground
		{x=5, y=-2, z=1},
		{x=5, y=-2, z=2},
		{x=5, y=-2, z=3},
		{x=5, y=-2, z=4},
		{x=5, y=-2, z=5},

		{x=4, y=-2, z=5},
		{x=3, y=-2, z=5},
		{x=2, y=-2, z=5},
		{x=1, y=-2, z=5},

		{x=-1, y=-2, z=5},
		{x=-2, y=-2, z=5},
		{x=-3, y=-2, z=5},
		{x=-4, y=-2, z=5},
		{x=-5, y=-2, z=5},

		{x=-5, y=-2, z=4},
		{x=-5, y=-2, z=3},
		{x=-5, y=-2, z=2},
		{x=-5, y=-2, z=1},

		{x=-5, y=-2, z=-1},
		{x=-5, y=-2, z=-2},
		{x=-5, y=-2, z=-3},
		{x=-5, y=-2, z=-4},
		{x=-5, y=-2, z=-5},

		{x=-4, y=-2, z=-5},
		{x=-3, y=-2, z=-5},
		{x=-2, y=-2, z=-5},
		{x=-1, y=-2, z=-5},

		{x=1, y=-2, z=-5},
		{x=2, y=-2, z=-5},
		{x=3, y=-2, z=-5},
		{x=4, y=-2, z=-5},
		{x=5, y=-2, z=-5},

		{x=5, y=-2, z=-4},
		{x=5, y=-2, z=-3},
		{x=5, y=-2, z=-2},
		{x=5, y=-2, z=-1},

		-- in each inner wall corner
		{x=3, y=-2, z=3},
		{x=3, y=-2, z=-3},
		{x=-3, y=-2, z=3},
		{x=-3, y=-2, z=-3}
	}},
	{{"group:black"},
	{
		-- walls and pillars are built from black, and there are some on ground.
		-- pillars
		{x=2, y=-1, z=2},
		{x=2, y=0, z=2},
		{x=2, y=1, z=2},

		{x=2, y=-1, z=-2},
		{x=2, y=0, z=-2},
		{x=2, y=1, z=-2},

		{x=-2, y=-1, z=2},
		{x=-2, y=0, z=2},
		{x=-2, y=1, z=2},

		{x=-2, y=-1, z=-2},
		{x=-2, y=0, z=-2},
		{x=-2, y=1, z=-2},

		-- exterior walls
		{x=6, y=-1, z=2},
		{x=6, y=0, z=2},
		{x=6, y=-1, z=3},
		{x=6, y=0, z=3},

		{x=6, y=-1, z=5},
		{x=6, y=0, z=5},
		{x=6, y=-1, z=6},
		{x=6, y=0, z=6},
		{x=5, y=-1, z=6},
		{x=5, y=0, z=6},

		{x=3, y=-1, z=6},
		{x=3, y=0, z=6},
		{x=2, y=-1, z=6},
		{x=2, y=0, z=6},

		{x=-2, y=-1, z=6},
		{x=-2, y=0, z=6},
		{x=-3, y=-1, z=6},
		{x=-3, y=0, z=6},

		{x=-5, y=-1, z=6},
		{x=-5, y=0, z=6},
		{x=-6, y=-1, z=6},
		{x=-6, y=0, z=6},
		{x=-6, y=-1, z=5},
		{x=-6, y=0, z=5},

		{x=-6, y=-1, z=3},
		{x=-6, y=0, z=3},
		{x=-6, y=-1, z=2},
		{x=-6, y=0, z=2},

		{x=-6, y=-1, z=-2},
		{x=-6, y=0, z=-2},
		{x=-6, y=-1, z=-3},
		{x=-6, y=0, z=-3},

		{x=-6, y=-1, z=-5},
		{x=-6, y=0, z=-5},
		{x=-6, y=-1, z=-6},
		{x=-6, y=0, z=-6},
		{x=-5, y=-1, z=-6},
		{x=-5, y=0, z=-6},

		{x=-3, y=-1, z=-6},
		{x=-3, y=0, z=-6},
		{x=-2, y=-1, z=-6},
		{x=-2, y=0, z=-6},

		{x=2, y=-1, z=-6},
		{x=2, y=0, z=-6},
		{x=3, y=-1, z=-6},
		{x=3, y=0, z=-6},

		{x=5, y=-1, z=-6},
		{x=5, y=0, z=-6},
		{x=6, y=-1, z=-6},
		{x=6, y=0, z=-6},
		{x=6, y=-1, z=-5},
		{x=6, y=0, z=-5},

		{x=6, y=-1, z=-3},
		{x=6, y=0, z=-3},
		{x=6, y=-1, z=-2},
		{x=6, y=0, z=-2},

		-- interior walls
		{x=4, y=-1, z=2},
		{x=4, y=0, z=2},
		{x=4, y=-1, z=3},
		{x=4, y=0, z=3},
		{x=4, y=-1, z=4},
		{x=4, y=0, z=4},
		{x=3, y=-1, z=4},
		{x=3, y=0, z=4},
		{x=2, y=-1, z=4},
		{x=2, y=0, z=4},

		{x=-2, y=-1, z=4},
		{x=-2, y=0, z=4},
		{x=-3, y=-1, z=4},
		{x=-3, y=0, z=4},
		{x=-4, y=-1, z=4},
		{x=-4, y=0, z=4},
		{x=-4, y=-1, z=3},
		{x=-4, y=0, z=3},
		{x=-4, y=-1, z=2},
		{x=-4, y=0, z=2},

		{x=-4, y=-1, z=-2},
		{x=-4, y=0, z=-2},
		{x=-4, y=-1, z=-3},
		{x=-4, y=0, z=-3},
		{x=-4, y=-1, z=-4},
		{x=-4, y=0, z=-4},
		{x=-3, y=-1, z=-4},
		{x=-3, y=0, z=-4},
		{x=-2, y=-1, z=-4},
		{x=-2, y=0, z=-4},

		{x=2, y=-1, z=-4},
		{x=2, y=0, z=-4},
		{x=3, y=-1, z=-4},
		{x=3, y=0, z=-4},
		{x=4, y=-1, z=-4},
		{x=4, y=0, z=-4},
		{x=4, y=-1, z=-3},
		{x=4, y=0, z=-3},
		{x=4, y=-1, z=-2},
		{x=4, y=0, z=-2},

		-- on the ground
		-- outer circle filler
		{x=6, y=-2, z=1},
		{x=6, y=-2, z=4},

		{x=4, y=-2, z=6},
		{x=1, y=-2, z=6},

		{x=-1, y=-2, z=6},
		{x=-4, y=-2, z=6},

		{x=-6, y=-2, z=4},
		{x=-6, y=-2, z=1},

		{x=-6, y=-2, z=-1},
		{x=-6, y=-2, z=-4},

		{x=-4, y=-2, z=-6},
		{x=-1, y=-2, z=-6},

		{x=1, y=-2, z=-6},
		{x=4, y=-2, z=-6},

		{x=6, y=-2, z=-4},
		{x=6, y=-2, z=-1},

		-- courtyard fill (next to yellow walkways)
		{x=4, y=-2, z=1},
		{x=3, y=-2, z=1},
		{x=2, y=-2, z=1},
		{x=1, y=-2, z=1},
		{x=1, y=-2, z=2},
		{x=1, y=-2, z=3},
		{x=1, y=-2, z=4},

		{x=-1, y=-2, z=4},
		{x=-1, y=-2, z=3},
		{x=-1, y=-2, z=2},
		{x=-1, y=-2, z=1},
		{x=-2, y=-2, z=1},
		{x=-3, y=-2, z=1},
		{x=-4, y=-2, z=1},

		{x=-4, y=-2, z=-1},
		{x=-3, y=-2, z=-1},
		{x=-2, y=-2, z=-1},
		{x=-1, y=-2, z=-1},
		{x=-1, y=-2, z=-2},
		{x=-1, y=-2, z=-3},
		{x=-1, y=-2, z=-4},

		{x=1, y=-2, z=-4},
		{x=1, y=-2, z=-3},
		{x=1, y=-2, z=-2},
		{x=1, y=-2, z=-1},
		{x=2, y=-2, z=-1},
		{x=3, y=-2, z=-1},
		{x=4, y=-2, z=-1}
	}},
	{{"group:purple"},
	{
		-- in central pillar and ontop of other pillars
		-- central pillar
		{x=0, y=1, z=0},
		{x=0, y=2, z=0},

		-- other pillars
		{x=2, y=2, z=2},
		{x=2, y=2, z=-2},
		{x=-2, y=2, z=2},
		{x=-2, y=2, z=-2}
	}},

}


local function has_position(tab, pos)
	for _,te in ipairs(tab) do
		if te.x == pos.x and te.y == pos.y and te.z == pos.z then
			return true
		end
	end
	return false
end

local function get_colour(nodespec)
	-- NOTE: function assumes tables contain a single string with a node group.
	if type(nodespec) == "table" then
		if nodespec[1] == "group:yellow" then
			return "yellow"
		elseif nodespec[1] == "group:red" then
			return "red"
		elseif nodespec[1] == "group:green" then
			return "green"
		elseif nodespec[1] == "group:black" then
			return "black"
		elseif nodespec[1] == "group:blue" then
			return "blue"
		elseif nodespec[1] == "group:purple" then
			return "purple"
		end
		print("[ERROR] Couldn't find colour from node group")
		return "unknown_group"
	else
		if nodespec == "world:yellow" or nodespec == "buildings:yellow" then
			return "yellow"
		elseif nodespec == "world:red" or nodespec == "buildings:red" then
			return "red"
		elseif nodespec == "world:green" or nodespec == "buildings:green" then
			return "green"
		elseif nodespec == "world:black" or nodespec == "buildings:black" then
			return "black"
		elseif nodespec == "world:blue" or nodespec == "buildings:blue" then
			return "blue"
		elseif nodespec == "world:purple" or nodespec == "buildings:purple" then
			return "purple"
		end
		print("[ERROR] Couldn't find colour from node name")
		return "unknown_name"
	end
end

local function check_schem_colour(schem, target, c, rx, ry, rz)
	local nodes = minetest.find_nodes_in_area({x=c.x-rx, y=c.y-ry, z=c.z-rz}, {x=c.x+rx, y=c.y+ry, z=c.z+rz}, target)
	for _,scp in ipairs(schem) do
		local a = {x=c.x+scp.x, y=c.y+scp.y, z=c.z+scp.z}
		if not has_position(nodes, a) then
			return false
		end
	end
	return true
end

local function check_for_altar1(top_green_pos)
	local radx = 4
	local rady = 2
	local radz = 4
	local center_pos = {x=top_green_pos.x, y=top_green_pos.y-2, z=top_green_pos.z}
	-- check if all required nodes are present
	for _,schem in ipairs(altar1) do
		if not check_schem_colour(schem[2], schem[1], center_pos, radx, rady, radz) then
			return false
		end
	end
	-- Check that we have sky above the whole thing.
	for _,yellowp in ipairs(altar1[2][2]) do
		local a = {x=center_pos.x+yellowp.x, y=center_pos.y+yellowp.y, z=center_pos.z+yellowp.z}
		if not buildings.sky_above(a) then
			return false
		end
	end
	for _,greenp in ipairs(altar1[3][2]) do
		local a = {x=center_pos.x+greenp.x, y=center_pos.y+greenp.y, z=center_pos.z+greenp.z}
		if not buildings.sky_above(a) then
			return false
		end
	end
	for _,blackp in ipairs(altar1[4][2]) do
		local a = {x=center_pos.x+blackp.x, y=center_pos.y+blackp.y, z=center_pos.z+blackp.z}
		if not buildings.sky_above(a) then
			return false
		end
	end
	-- If we got here, then the altar is valid. Activate it!
	for _,schem in ipairs(altar1) do
		if schem[1] ~= "air" then
			local colour = get_colour(schem[1])
			-- Convet to (temporarily?) indestructible blocks.
			for _,scp in ipairs(schem[2]) do
				local a = {x=center_pos.x+scp.x, y=center_pos.y+scp.y, z=center_pos.z+scp.z}
				minetest.set_node(a, {name = "buildings:" .. colour})
			end
		end
	end
	-- Set top green block to altar main block (timer for particle effects)
	minetest.set_node(top_green_pos, {name = "buildings:altar1"})
	local tm = minetest.get_node_timer(top_green_pos)
	tm:start(3)
	return true
end

local function build_altar1(top_green_pos)
	-- debug function..
	local radx = 4
	local rady = 2
	local radz = 4
	local center_pos = {x=top_green_pos.x, y=top_green_pos.y-2, z=top_green_pos.z}
	for _,schem in ipairs(altar1) do
		if schem[1] ~= "air" then
			local colour = get_colour(schem[1])
			for _,scp in ipairs(schem[2]) do
				local a = {x=center_pos.x+scp.x, y=center_pos.y+scp.y, z=center_pos.z+scp.z}
				minetest.set_node(a, {name = "world:" .. colour})
			end
		end
	end
	return true
end

local function check_for_altar2(top_green_pos)
	local radx = 4
	local rady = 2
	local radz = 4
	local center_pos = {x=top_green_pos.x, y=top_green_pos.y-2, z=top_green_pos.z}
	-- check if all required nodes are present
	for _,schem in ipairs(altar2) do
		if not check_schem_colour(schem[2], schem[1], center_pos, radx, rady, radz) then
			return false
		end
	end
	-- Check that we have sky above the whole thing.
	for _,redp in ipairs(altar2[1][2]) do
		local a = {x=center_pos.x+redp.x, y=center_pos.y+redp.y, z=center_pos.z+redp.z}
		if not buildings.sky_above(a) then
			return false
		end
	end
	for _,yellowp in ipairs(altar2[2][2]) do
		local a = {x=center_pos.x+yellowp.x, y=center_pos.y+yellowp.y, z=center_pos.z+yellowp.z}
		if not buildings.sky_above(a) then
			return false
		end
	end
	for _,greenp in ipairs(altar2[3][2]) do
		local a = {x=center_pos.x+greenp.x, y=center_pos.y+greenp.y, z=center_pos.z+greenp.z}
		if not buildings.sky_above(a) then
			return false
		end
	end
	for _,blackp in ipairs(altar2[5][2]) do
		local a = {x=center_pos.x+blackp.x, y=center_pos.y+blackp.y, z=center_pos.z+blackp.z}
		if not buildings.sky_above(a) then
			return false
		end
	end
	-- If we got here, then the altar is valid. Activate it!
	for _,schem in ipairs(altar2) do
		if schem[1] ~= "air" then
			local colour = get_colour(schem[1])
			-- Convet to (temporarily?) indestructible blocks.
			for _,scp in ipairs(schem[2]) do
				local a = {x=center_pos.x+scp.x, y=center_pos.y+scp.y, z=center_pos.z+scp.z}
				minetest.set_node(a, {name = "buildings:" .. colour})
			end
		end
	end
	-- Set top green block to altar main block (timer for particle effects)
	minetest.set_node(top_green_pos, {name = "buildings:altar2"})
	local tm = minetest.get_node_timer(top_green_pos)
	tm:start(3)
	return true
end

local function build_altar2(top_green_pos)
	-- debug function..
	local radx = 4
	local rady = 2
	local radz = 4
	local center_pos = {x=top_green_pos.x, y=top_green_pos.y-2, z=top_green_pos.z}
	for _,schem in ipairs(altar2) do
		if schem[1] ~= "air" then
			local colour = get_colour(schem[1])
			for _,scp in ipairs(schem[2]) do
				local a = {x=center_pos.x+scp.x, y=center_pos.y+scp.y, z=center_pos.z+scp.z}
				minetest.set_node(a, {name = "world:" .. colour})
			end
		end
	end
	return true
end

local function check_for_altar3(top_green_pos)
	local radx = 7
	local rady = 3
	local radz = 7
	local center_pos = {x=top_green_pos.x, y=top_green_pos.y-3, z=top_green_pos.z}
	-- check if all required nodes are present
	for _,schem in ipairs(altar3) do
		if not check_schem_colour(schem[2], schem[1], center_pos, radx, rady, radz) then
			return false
		end
	end
	-- Check that we have sky above the whole thing.
	for _,yellowp in ipairs(altar3[1][2]) do
		local a = {x=center_pos.x+yellowp.x, y=center_pos.y+yellowp.y, z=center_pos.z+yellowp.z}
		if not buildings.sky_above(a) then
			return false
		end
	end
	for _,greenp in ipairs(altar3[2][2]) do
		local a = {x=center_pos.x+greenp.x, y=center_pos.y+greenp.y, z=center_pos.z+greenp.z}
		if not buildings.sky_above(a) then
			return false
		end
	end
	-- for other colours, only check if ypos == -2
	for _,bluep in ipairs(altar3[3][2]) do
		if bluep.y == -2 then
			local a = {x=center_pos.x+bluep.x, y=center_pos.y+bluep.y, z=center_pos.z+bluep.z}
			if not buildings.sky_above(a) then
				return false
			end
		end
	end
	for _,blackp in ipairs(altar3[4][2]) do
		if blackp.y == -2 then
			local a = {x=center_pos.x+blackp.x, y=center_pos.y+blackp.y, z=center_pos.z+blackp.z}
			if not buildings.sky_above(a) then
				return false
			end
		end
	end

	-- next, check the walls.. a bit more complicated..
	for _,blackp in ipairs(altar3[4][2]) do
		if blackp.y == 0 then
			if blackp.x > 2 or blackp.x < -2 or blackp.z > 2 or blackp.z < -2 then
				local a = {x=center_pos.x+blackp.x, y=center_pos.y+blackp.y, z=center_pos.z+blackp.z}
				if not buildings.sky_above(a) then
					return false
				end
			end
		end
	end

	-- finally, check that the purple on each pillar has sky above it
	for _,purplep in ipairs(altar3[5][2]) do
		if purplep.y == 2 and purplep.x ~= 0 then
			local a = {x=center_pos.x+purplep.x, y=center_pos.y+purplep.y, z=center_pos.z+purplep.z}
			if not buildings.sky_above(a) then
				return false
			end
		end
	end

	-- If we got here, then the altar is valid. Activate it!
	for _,schem in ipairs(altar3) do
		if schem[1] ~= "air" then
			local colour = get_colour(schem[1])
			-- Convet to (temporarily?) indestructible blocks.
			for _,scp in ipairs(schem[2]) do
				local a = {x=center_pos.x+scp.x, y=center_pos.y+scp.y, z=center_pos.z+scp.z}
				minetest.set_node(a, {name = "buildings:" .. colour})
			end
		end
	end
	-- Set top green block to altar main block (timer for particle effects)
	minetest.set_node(top_green_pos, {name = "buildings:altar3"})
	local tm = minetest.get_node_timer(top_green_pos)
	tm:start(3)
	return true
end

local function build_altar3(top_green_pos)
	-- debug function..
	local radx = 7
	local rady = 3
	local radz = 7
	local center_pos = {x=top_green_pos.x, y=top_green_pos.y-3, z=top_green_pos.z}
	for _,schem in ipairs(altar3) do
		if schem[1] ~= "air" then
			local colour = get_colour(schem[1])
			for _,scp in ipairs(schem[2]) do
				local a = {x=center_pos.x+scp.x, y=center_pos.y+scp.y, z=center_pos.z+scp.z}
				minetest.set_node(a, {name = "world:" .. colour})
			end
		end
	end
	return true
end

local function build_ion_cannon(top_green_pos)
	-- debug function..
	local radx = 3
	local rady = 2
	local radz = 3
	local center_pos = {x=top_green_pos.x, y=top_green_pos.y-1, z=top_green_pos.z}
	for _,schem in ipairs(ion_cannon) do
		if schem[1] ~= "air" then
			local colour = get_colour(schem[1])
			for _,scp in ipairs(schem[2]) do
				local a = {x=center_pos.x+scp.x, y=center_pos.y+scp.y, z=center_pos.z+scp.z}
				minetest.set_node(a, {name = "world:" .. colour})
			end
		end
	end
	return true
end

buildings.check_for_ion_cannon = function(bottom_center_pos)
	local radx = 3
	local rady = 2
	local radz = 3
	local center_pos = {x=bottom_center_pos.x, y=bottom_center_pos.y+2, z=bottom_center_pos.z}

	-- check if all required nodes are present
	for _,schem in ipairs(ion_cannon) do
		if not check_schem_colour(schem[2], schem[1], center_pos, radx, rady, radz) then
			return false
		end
	end
	-- Check that we have sky above the whole thing.
	for _,yellowp in ipairs(ion_cannon[1][2]) do
		local a = {x=center_pos.x+yellowp.x, y=center_pos.y+yellowp.y, z=center_pos.z+yellowp.z}
		if not buildings.sky_above(a) then
			return false
		end
	end
	for _,blackp in ipairs(ion_cannon[3][2]) do
		local a = {x=center_pos.x+blackp.x, y=center_pos.y+blackp.y, z=center_pos.z+blackp.z}
		if not buildings.sky_above(a) then
			return false
		end
	end
	for _,bluep in ipairs(ion_cannon[4][2]) do
		local a = {x=center_pos.x+bluep.x, y=center_pos.y+bluep.y, z=center_pos.z+bluep.z}
		if not buildings.sky_above(a) then
			return false
		end
	end
	-- Finally, check that we have air above the central pillar
	if not buildings.sky_above({x=center_pos.x, y=center_pos.y+1, z=center_pos.z}) then
		return false
	end

	-- Yep, thats an ion cannon alright
	return true
end


buildings.check_on_green = function(pos)
	-- check if this is some kind of building that should be activated
	if check_for_altar1(pos) then
		return true
	end
	if check_for_altar2(pos) then
		return true
	end
	if check_for_altar3(pos) then
		return true
	end

	return false
end


local w = minetest.get_modpath("buildings")

dofile(w .. "/blocks.lua")
