
tdls_staffs = {}
local passive_staffs, player_mana, cooldowns = {}, {}, {}
local MAX_MANA, MANA_REGEN_RATE, MANA_REGEN_INTERVAL = 100, 5, 1
local regen_accum, hb_ok = 0, rawget(_G, "hb") ~= nil

local function now() return minetest.get_us_time() / 1e6 end

local function shallow_copy(t)
  if type(t) ~= "table" then return t end
  local n = {}
  for k, v in pairs(t) do n[k] = v end
  return n
end

local function hud_register()
  if hb_ok and hb.register_hudbar then
    hb.register_hudbar("mana", 0x0000ff, "Mana", {bar = "hudbars_bar_blue.png", icon = "mana_icon.png"}, MAX_MANA, MAX_MANA, false)
  end
end

local function hud_init(player, value, max)
  if hb_ok and hb.init_hudbar then
    hb.init_hudbar(player, "mana", value, max or MAX_MANA)
  end
end

local function hud_change(player, value, max)
  if hb_ok and hb.change_hudbar then
    hb.change_hudbar(player, "mana", value, max or MAX_MANA)
  end
end

minetest.register_on_mods_loaded(hud_register)

local function is_on_cd(player, itemname, action)
  local name = player and player:get_player_name()
  if not name then return false end
  local key = itemname .. ":" .. action
  local t = cooldowns[name]
  return t and t[key] and t[key] > now()
end

local function start_cd(player, itemname, action, seconds)
  if not seconds or seconds <= 0 then return end
  local name = player and player:get_player_name()
  if not name then return end
  cooldowns[name] = cooldowns[name] or {}
  cooldowns[name][itemname .. ":" .. action] = now() + seconds
end

function tdls_staffs.get_mana(player)
  if not player or not player:is_player() then return MAX_MANA end
  return player_mana[player:get_player_name()] or MAX_MANA
end

function tdls_staffs.set_mana(player, amount)
  if not player or not player:is_player() then return end
  local name = player:get_player_name()
  local new_mana = math.min(math.max(amount, 0), MAX_MANA)
  player_mana[name] = new_mana
  hud_change(player, new_mana, MAX_MANA)
end

function tdls_staffs.use_mana(player, cost)
  if not player or not player:is_player() then return false end
  local current = tdls_staffs.get_mana(player)
  cost = cost or 0
  if current >= cost then
    tdls_staffs.set_mana(player, current - cost)
    return true
  end
  minetest.chat_send_player(player:get_player_name(), "Not enough mana!")
  return false
end

function tdls_staffs.register_staff(name, def)
  if not name or type(def) ~= "table" then return end
  minetest.register_tool(name, {
    description = def.description or "Mystic Staff",
    inventory_image = def.inventory_image or "default_tool_steelaxe.png",
    on_use = function(itemstack, user, pointed_thing)
      if def.left_cd and is_on_cd(user, name, "left") then
        minetest.chat_send_player(user:get_player_name(), "Ability is on cooldown.")
        return itemstack
      end
      if def.on_leftclick and tdls_staffs.use_mana(user, def.left_mana or 0) then
        pcall(def.on_leftclick, user, pointed_thing)
        if def.left_cd then start_cd(user, name, "left", def.left_cd) end
      end
      return itemstack
    end,
    on_place = function(itemstack, user, pointed_thing)
      if def.right_cd and is_on_cd(user, name, "right") then
        minetest.chat_send_player(user:get_player_name(), "Ability is on cooldown.")
        return itemstack
      end
      if def.on_rightclick and tdls_staffs.use_mana(user, def.right_mana or 0) then
        pcall(def.on_rightclick, user, pointed_thing)
        if def.right_cd then start_cd(user, name, "right", def.right_cd) end
      end
      return itemstack
    end
  })
  if def.passive then
    passive_staffs[name] = { fn = def.passive, cost = def.passive_mana or 0 }
  end
end

minetest.register_on_joinplayer(function(player)
  player_mana[player:get_player_name()] = MAX_MANA
  hud_init(player, MAX_MANA, MAX_MANA)
end)

minetest.register_on_leaveplayer(function(player)
  local name = player and player:get_player_name()
  if not name then return end
  player_mana[name], cooldowns[name] = nil, nil
end)

minetest.register_globalstep(function(dtime)
  regen_accum = regen_accum + dtime
  if regen_accum >= MANA_REGEN_INTERVAL then
    regen_accum = regen_accum - MANA_REGEN_INTERVAL
    for _, player in ipairs(minetest.get_connected_players()) do
      local current = tdls_staffs.get_mana(player)
      if current < MAX_MANA then
        tdls_staffs.set_mana(player, math.min(current + MANA_REGEN_RATE, MAX_MANA))
      end
      local wield = player:get_wielded_item():get_name()
      local passive = passive_staffs[wield]
      if passive and passive.fn and (passive.cost == 0 or tdls_staffs.use_mana(player, passive.cost)) then
        pcall(passive.fn, player, MANA_REGEN_INTERVAL)
      end
    end
  end
end)

local burning = {}
local burn_tick = 0
local burn_interval = 0.5
local burn_damage = 1
local burn_duration = 5
local bolt_speed = 25

minetest.register_on_leaveplayer(function(player)
  for obj in pairs(burning) do
    if obj and obj:is_player() and obj:get_player_name() == player:get_player_name() then
      burning[obj] = nil
    end
  end
end)

minetest.register_entity("tdls_staffs:fire_bolt", {
  initial_properties = {
    physical = false,
    collide_with_objects = true,
    collisionbox = {0, 0, 0, 0, 0, 0},
    visual = "sprite",
    visual_size = {x = 0.5, y = 0.5},
    textures = {"fire_core.png"},
    pointable = false,
    static_save = false,
    hp_max = 1
  },
  _life = 0,
  _owner = "",
  on_step = function(self, dtime)
    self._life = self._life + dtime
    if self._life > 4 then
      self.object:remove()
      return
    end
    local pos = self.object:get_pos()
    if not pos then
      self.object:remove()
      return
    end
    for _, obj in ipairs(minetest.get_objects_inside_radius(pos, 1)) do
      if obj ~= self.object then
        local is_owner = obj:is_player() and obj:get_player_name() == self._owner
        if not is_owner then
          local owner = nil
          if self._owner and self._owner ~= "" then
            owner = minetest.get_player_by_name(self._owner)
          end
          burning[obj] = { expires = now() + burn_duration, puncher = owner }
          minetest.add_particle({ pos = pos, velocity = {x = 0, y = 1, z = 0}, expirationtime = 0.2, size = 4, texture = "fire_basic_flame.png", glow = 8 })
          self.object:remove()
          break
        end
      end
    end
  end
})

local function shoot_bolt(player)
  if not player or not player:is_player() then return end
  local eye = vector.add(player:get_pos(), {x = 0, y = 1.5, z = 0})
  local dir = player:get_look_dir()
  local obj = minetest.add_entity(eye, "tdls_staffs:fire_bolt")
  if not obj then return end
  obj:set_velocity(vector.multiply(dir, bolt_speed))
  obj:set_yaw(minetest.dir_to_yaw(dir))
  local ent = obj:get_luaentity()
  if ent then ent._owner = player:get_player_name() end
end

minetest.register_globalstep(function(dtime)
  burn_tick = burn_tick + dtime
  if burn_tick >= burn_interval then
    for obj, data in pairs(burning) do
      if not obj or not obj:get_pos() then
        burning[obj] = nil
      elseif now() >= data.expires then
        burning[obj] = nil
      else
        if obj.punch and data.puncher and data.puncher:is_player() then
          obj:punch(data.puncher, 1.0, { full_punch_interval = 1.0, damage_groups = { fleshy = burn_damage } }, nil)
        elseif obj.get_hp and obj.set_hp then
          local hp = obj:get_hp()
          if hp and hp > 0 then obj:set_hp(hp - burn_damage) end
        end
        local pos = obj:get_pos()
        if pos then
          minetest.add_particle({ pos = vector.add(pos, {x = 0, y = 1, z = 0}), velocity = {x = 0, y = 2, z = 0}, acceleration = {x = 0, y = 1, z = 0}, expirationtime = 0.4, size = 3, texture = "fire_basic_flame.png", glow = 8 })
        end
      end
    end
    burn_tick = 0
  end
end)

local beam_length = 12
local beam_duration = 4

local function place_lava_line(player)
  if not player or not player:is_player() then return end
  local eye = vector.add(player:get_pos(), {x = 0, y = 1.5, z = 0})
  local dir = player:get_look_dir()
  local placed = {}
  for i = 1, beam_length do
    local ip = vector.round(vector.add(eye, vector.multiply(dir, i)))
    local nd = minetest.get_node_or_nil(ip)
    if nd and minetest.registered_nodes[nd.name] then
      if minetest.is_protected(ip, player:get_player_name()) then break end
      if nd.name == "air" or minetest.get_item_group(nd.name, "liquid") ~= 0 then
        table.insert(placed, { pos = vector.new(ip), old = nd })
        minetest.set_node(ip, { name = "default:lava_flowing" })
      end
    end
  end
  minetest.after(beam_duration, function()
    for _, rec in ipairs(placed) do
      if minetest.get_node(rec.pos).name == "default:lava_flowing" then
        minetest.set_node(rec.pos, rec.old or { name = "air" })
      end
    end
  end)
end

local function is_hot_node(n)
  local name = n and n.name or ""
  return name == "fire:basic_flame" or name == "fire:permanent_flame" or name:find("lava", 1, true)
end

local function heal_in_fire(player)
  if not player or not player:is_player() then return end
  local p = player:get_pos()
  local feet = minetest.get_node_or_nil({ x = p.x, y = p.y, z = p.z })
  local head = minetest.get_node_or_nil({ x = p.x, y = p.y + 1, z = p.z })
  if is_hot_node(feet) or is_hot_node(head) then
    local props = player:get_properties() or {}
    local maxhp = props.hp_max or 20
    local hp = player:get_hp()
    if hp < maxhp then
      player:set_hp(math.min(maxhp, hp + 2))
    end
  end
  if feet and (feet.name == "fire:basic_flame" or feet.name == "fire:permanent_flame") then
    minetest.set_node({ x = p.x, y = p.y, z = p.z }, { name = "air" })
  end
end

tdls_staffs.register_staff("tdls_staffs:staff_fire", {
  description = "Fire Staff",
  inventory_image = "magic_staff_fire.png",
  left_mana = 15,
  right_mana = 25,
  passive_mana = 0,
  on_leftclick = function(player, pointed_thing) shoot_bolt(player) end,
  on_rightclick = function(player, pointed_thing) place_lava_line(player) end,
  passive = function(player, dtime) heal_in_fire(player, dtime) end
})

minetest.register_craft({
  output = "tdls_staffs:staff_fire",
  recipe = {
    {"", "tdls_staffs:fire_core", ""},
    {"", "default:stick", ""},
    {"", "default:stick", ""}
  }
})

minetest.register_craftitem("tdls_staffs:magic_core", {
  description = "Magic Core",
  inventory_image = "magic_core.png",
  stack_max = 99
})

minetest.register_craft({
  output = "tdls_staffs:magic_core",
  recipe = {
    {"", "ethereal:crystal_spike", ""},
    {"ethereal:crystal_spike", "default:mese_shard", "ethereal:crystal_spike"},
    {"", "ethereal:crystal_spike", ""}
  }
})

minetest.register_craftitem("tdls_staffs:fire_core", {
  description = "Fire Core",
  inventory_image = "fire_core.png",
  stack_max = 99
})

minetest.register_craft({
  output = "tdls_staffs:fire_core",
  recipe = {
    {"", "ethereal:crystal_spike", ""},
    {"ethereal:fire_flower", "tdls_staffs:magic_core", "ethereal:fire_flower"},
    {"", "ethereal:crystal_spike", ""}
  }
})

local stunned = {}
local STUN_DURATION = 10
local ICE_DAMAGE = 5

minetest.register_on_leaveplayer(function(player)
  for obj, data in pairs(stunned) do
    if obj and obj:is_player() and obj:get_player_name() == player:get_player_name() then
      if data.saved_phys then obj:set_physics_override(data.saved_phys) end
      stunned[obj] = nil
    end
  end
end)

minetest.register_globalstep(function(dtime)
  for obj, data in pairs(stunned) do
    if not obj or not obj:get_pos() then
      stunned[obj] = nil
    elseif now() >= data.expires then
      if obj:is_player() and data.saved_phys then obj:set_physics_override(data.saved_phys) end
      stunned[obj] = nil
    else
      if obj:is_player() then
        if not data.saved_phys then data.saved_phys = obj:get_physics_override() or {speed=1, jump=1, gravity=1} end
        obj:set_physics_override({ speed = 0, jump = 0, gravity = data.saved_phys.gravity or 1 })
      else
        if obj.set_velocity then
          local v = obj:get_velocity() or {x=0,y=0,z=0}
          obj:set_velocity({x=0, y=v.y, z=0})
        end
      end
    end
  end
end)

local ice_shard_speed = 30

minetest.register_entity("tdls_staffs:ice_shard", {
  initial_properties = {
    physical = false,
    collide_with_objects = true,
    collisionbox = {0, 0, 0, 0, 0, 0},
    visual = "sprite",
    visual_size = {x = 0.5, y = 0.5},
    textures = {"ice_core.png"},
    pointable = false,
    static_save = false,
    hp_max = 1
  },
  _life = 0,
  _owner = "",
  on_step = function(self, dtime)
    self._life = self._life + dtime
    if self._life > 4 then
      self.object:remove()
      return
    end
    local pos = self.object:get_pos()
    if not pos then
      self.object:remove()
      return
    end
    for _, obj in ipairs(minetest.get_objects_inside_radius(pos, 1)) do
      if obj ~= self.object then
        local is_owner = obj:is_player() and obj:get_player_name() == self._owner
        local is_valid = obj:is_player() or obj:get_luaentity()
        if is_valid and not is_owner then
          local puncher = nil
          if self._owner and self._owner ~= "" then puncher = minetest.get_player_by_name(self._owner) end
          if obj.punch and puncher and puncher:is_player() then
            obj:punch(puncher, 1.0, { full_punch_interval = 1.0, damage_groups = { fleshy = ICE_DAMAGE } }, nil)
          elseif obj.get_hp and obj.set_hp then
            local hp = obj:get_hp()
            if hp and hp > 0 then obj:set_hp(hp - ICE_DAMAGE) end
          end
          stunned[obj] = { expires = now() + STUN_DURATION }
          minetest.add_particle({ pos = pos, velocity = {x = 0, y = 1, z = 0}, expirationtime = 0.3, size = 4, texture = "default_snow.png", glow = 3 })
          self.object:remove()
          break
        end
      end
    end
  end
})

local function shoot_ice_shard(player)
  if not player or not player:is_player() then return end
  local eye = vector.add(player:get_pos(), {x = 0, y = 1.5, z = 0})
  local dir = player:get_look_dir()
  local obj = minetest.add_entity(eye, "tdls_staffs:ice_shard")
  if not obj then return end
  obj:set_velocity(vector.multiply(dir, ice_shard_speed))
  obj:set_acceleration({x = 0, y = 0, z = 0})
  obj:set_yaw(minetest.dir_to_yaw(dir))
  local ent = obj:get_luaentity()
  if ent then ent._owner = player:get_player_name() end
end

local frost_active = {}
local MANA_PER_SEC = 5
local FROST_REVERT_TIME = 2

local function frost_place_and_revert(pos, old_node)
  minetest.set_node(pos, { name = "default:ice" })
  minetest.after(FROST_REVERT_TIME, function()
    if minetest.get_node(pos).name == "default:ice" then
      minetest.set_node(pos, old_node or { name = "air" })
    end
  end)
end

local function frost_toggle(player)
  local name = player:get_player_name()
  frost_active[name] = not frost_active[name]
  minetest.chat_send_player(name, frost_active[name] and "Frost Sprint activated!" or "Frost Sprint deactivated.")
end

minetest.register_on_leaveplayer(function(player)
  local name = player and player:get_player_name()
  if name then frost_active[name] = nil end
end)

local frost_accum = 0

minetest.register_globalstep(function(dtime)
  frost_accum = frost_accum + dtime
  if frost_accum < 1 then return end
  frost_accum = 0
  for _, player in ipairs(minetest.get_connected_players()) do
    local name = player:get_player_name()
    if frost_active[name] then
      local wield = player:get_wielded_item()
      if not wield or wield:is_empty() or wield:get_name() ~= "tdls_staffs:staff_ice" then
        frost_active[name] = false
        minetest.chat_send_player(name, "Frost Sprint deactivated (not wielding Ice Staff).")
      else
        if not tdls_staffs.use_mana(player, MANA_PER_SEC) then
          frost_active[name] = false
          minetest.chat_send_player(name, "Out of mana. Frost Sprint stopped.")
        else
          local pos = player:get_pos()
          local under = vector.round({x = pos.x, y = pos.y - 1, z = pos.z})
          if not minetest.is_protected(under, name) then
            local old = minetest.get_node_or_nil(under)
            if old and minetest.registered_nodes[old.name] then
              if old.name ~= "default:ice" then
                frost_place_and_revert(under, shallow_copy(old))
              end
            end
          end
        end
      end
    end
  end
end)

local ice_speed_state = {}
local ICE_SPEED_MULT = 1.6
local ICE_SPEED_DELAY = 1.5

local function passive_ice_speed(player, dtime)
  local name = player:get_player_name()
  local p = player:get_pos()
  if not p then return end
  local under = minetest.get_node_or_nil({x = p.x, y = p.y - 1, z = p.z})
  local feet = minetest.get_node_or_nil({x = p.x, y = p.y, z = p.z})
  local on_ice = (under and under.name == "default:ice") or (feet and feet.name == "default:ice")
  local state = ice_speed_state[name]
  if on_ice then
    if not state then
      ice_speed_state[name] = { saved_phys = player:get_physics_override() or {speed=1, jump=1, gravity=1}, restore_at = nil }
    elseif state.restore_at then
      state.restore_at = nil
    end
    local saved = ice_speed_state[name].saved_phys
    player:set_physics_override({
      speed = (saved.speed or 1) * ICE_SPEED_MULT,
      jump = saved.jump or 1,
      gravity = saved.gravity or 1
    })
  else
    if state and not state.restore_at then
      state.restore_at = minetest.get_gametime() + ICE_SPEED_DELAY
    end
    if state and state.restore_at and minetest.get_gametime() >= state.restore_at then
      player:set_physics_override(state.saved_phys)
      ice_speed_state[name] = nil
    end
  end
end

minetest.register_on_leaveplayer(function(player)
  local name = player and player:get_player_name()
  if name and ice_speed_state[name] then
    player:set_physics_override(ice_speed_state[name].saved_phys)
    ice_speed_state[name] = nil
  end
end)

tdls_staffs.register_staff("tdls_staffs:staff_ice", {
  description = "Ice Staff",
  inventory_image = "magic_staff_ice.png",
  left_mana = 15,
  left_cd = 2.0,
  right_mana = 0,
  right_cd = 0,
  on_leftclick = function(player, pointed_thing) shoot_ice_shard(player) end,
  on_rightclick = function(player, pointed_thing) frost_toggle(player) end,
  passive = function(player, dtime) passive_ice_speed(player, dtime) end
})

minetest.register_craftitem("tdls_staffs:ice_core", {
  description = "Ice Core",
  inventory_image = "ice_core.png",
  stack_max = 99
})

minetest.register_craft({
  output = "tdls_staffs:ice_core",
  recipe = {
    {"", "default:ice", ""},
    {"default:ice", "tdls_staffs:magic_core", "default:ice"},
    {"", "default:ice", ""}
  }
})

minetest.register_craft({
  output = "tdls_staffs:staff_ice",
  recipe = {
    {"", "tdls_staffs:ice_core", ""},
    {"", "default:stick", ""},
    {"", "default:stick", ""}
  }
})

local AIR_FLIGHT_SPEED = 1000
local AIR_FLIGHT_MANA_PER_SEC = 10
local WIND_BOMB_SPEED = 25
local WIND_BOMB_LIFETIME = 5
local WIND_DIRECT_STRENGTH = 25
local WIND_AREA_STRENGTH = 20
local WIND_AREA_RADIUS = 1.5

local air_flight = {}

local function air_restore_phys(player, name)
  local st = air_flight[name]
  if st and st.saved_phys then player:set_physics_override(st.saved_phys) end
end

local function air_deactivate(player, msg)
  if not player or not player:is_player() then return end
  local name = player:get_player_name()
  if not name then return end
  if air_flight[name] and air_flight[name].active then
    if player.set_velocity then player:set_velocity({x=0,y=0,z=0}) end
    air_restore_phys(player, name)
    air_flight[name] = nil
    if msg then minetest.chat_send_player(name, msg) end
  end
end

local function air_activate(player)
  local name = player:get_player_name()
  if not name then return end
  if not air_flight[name] then
    air_flight[name] = { active = true, saved_phys = player:get_physics_override() or {speed=1, jump=1, gravity=1} }
  else
    air_flight[name].active = true
  end
  player:set_physics_override({
    speed = air_flight[name].saved_phys.speed or 1,
    jump = air_flight[name].saved_phys.jump or 1,
    gravity = 0
  })
  minetest.chat_send_player(name, "Air Flight activated!")
end

local function air_flight_toggle(player)
  if not player or not player:is_player() then return end
  local name = player:get_player_name()
  local st = air_flight[name]
  if st and st.active then
    air_deactivate(player, "Air Flight deactivated.")
  else
    air_activate(player)
  end
end

local air_accum = 0

minetest.register_globalstep(function(dtime)
  air_accum = air_accum + dtime
  for _, player in ipairs(minetest.get_connected_players()) do
    local name = player:get_player_name()
    local st = air_flight[name]
    if st and st.active then
      local wield = player:get_wielded_item()
      if not wield or wield:is_empty() or wield:get_name() ~= "tdls_staffs:staff_air" then
        air_deactivate(player, "Air Flight deactivated (not wielding Air Staff).")
      else
        local dir = player:get_look_dir()
        local v = vector.multiply(dir, AIR_FLIGHT_SPEED)
        if player.set_velocity then
          player:set_velocity(v)
        else
          local cur = player:get_velocity() or {x=0,y=0,z=0}
          player:add_velocity({x = v.x - cur.x, y = v.y - cur.y, z = v.z - cur.z})
        end
      end
    end
  end
  if air_accum >= 1 then
    for _, player in ipairs(minetest.get_connected_players()) do
      local name = player:get_player_name()
      local st = air_flight[name]
      if st and st.active then
        if not tdls_staffs.use_mana(player, AIR_FLIGHT_MANA_PER_SEC) then
          air_deactivate(player, "Out of mana. Air Flight stopped.")
        end
      end
    end
    air_accum = 0
  end
end)

minetest.register_on_leaveplayer(function(player)
  local name = player and player:get_player_name()
  if name and air_flight[name] then
    air_restore_phys(player, name)
    air_flight[name] = nil
  end
end)

minetest.register_on_player_hpchange(function(player, hp_change, reason)
  if hp_change < 0 and reason and reason.type == "fall" then
    local w = player:get_wielded_item()
    if w and not w:is_empty() and w:get_name() == "tdls_staffs:staff_air" then
      return 0
    end
  end
  return hp_change
end, true)

local function fling_up(obj, strength)
  if not obj then return end
  if obj.add_velocity then
    obj:add_velocity({x = 0, y = strength, z = 0})
  elseif obj.set_velocity then
    local v = obj:get_velocity() or {x=0,y=0,z=0}
    obj:set_velocity({x = v.x, y = v.y + strength, z = v.z})
  end
end

local function wind_aoe_fling(pos, strength)
  for _, obj in ipairs(minetest.get_objects_inside_radius(pos, WIND_AREA_RADIUS)) do
    if obj:is_player() or obj:get_luaentity() then
      fling_up(obj, strength)
    end
  end
  minetest.add_particle({ pos = pos, velocity = {x=0,y=2,z=0}, expirationtime = 0.5, size = 6, texture = "magic_wind_puff.png", glow = 5 })
  minetest.sound_play("magic_wind_blast", {pos = pos, gain = 0.7, max_hear_distance = 16}, true)
end

minetest.register_entity("tdls_staffs:wind_bomb", {
  initial_properties = {
    physical = false,
    collide_with_objects = true,
    collisionbox = {0,0,0, 0,0,0},
    visual = "sprite",
    visual_size = {x = 0.6, y = 0.6},
    textures = {"air_core.png"},
    pointable = false,
    static_save = false,
    hp_max = 1
  },
  _owner = "",
  _life = 0,
  on_step = function(self, dtime)
    self._life = self._life + dtime
    if self._life > WIND_BOMB_LIFETIME then
      self.object:remove()
      return
    end
    local pos = self.object:get_pos()
    if not pos then
      self.object:remove()
      return
    end
    for _, obj in ipairs(minetest.get_objects_inside_radius(pos, 1)) do
      if obj ~= self.object then
        local is_owner = obj:is_player() and obj:get_player_name() == self._owner
        local valid = obj:is_player() or obj:get_luaentity()
        if valid and not is_owner then
          fling_up(obj, WIND_DIRECT_STRENGTH)
          minetest.add_particle({ pos = pos, velocity = {x=0,y=2,z=0}, expirationtime = 0.3, size = 4, texture = "air_core.png", glow = 4 })
          minetest.sound_play("magic_wind_hit", {pos = pos, gain = 0.5, max_hear_distance = 16}, true)
          self.object:remove()
          return
        end
      end
    end
    local node = minetest.get_node_or_nil(pos)
    if node and minetest.registered_nodes[node.name] then
      local def = minetest.registered_nodes[node.name]
      if def.walkable then
        wind_aoe_fling(pos, WIND_AREA_STRENGTH)
        self.object:remove()
        return
      end
    end
  end
})

local function shoot_wind_bomb(player)
  if not player or not player:is_player() then return end
  local eye = vector.add(player:get_pos(), {x=0, y=1.5, z=0})
  local dir = player:get_look_dir()
  local obj = minetest.add_entity(eye, "tdls_staffs:wind_bomb")
  if not obj then return end
  obj:set_velocity(vector.multiply(dir, WIND_BOMB_SPEED))
  obj:set_acceleration({x=0, y=0, z=0})
  obj:set_yaw(minetest.dir_to_yaw(dir))
  local ent = obj:get_luaentity()
  if ent then ent._owner = player:get_player_name() end
end

tdls_staffs.register_staff("tdls_staffs:staff_air", {
  description = "Air Staff",
  inventory_image = "magic_staff_air.png",
  left_mana = 0,
  left_cd = 0.2,
  on_leftclick = function(player, _) air_flight_toggle(player) end,
  right_mana = 15,
  right_cd = 1.0,
  on_rightclick = function(player, _) shoot_wind_bomb(player) end,
  passive_mana = 0,
  passive = function(player, dtime) end
})

minetest.register_craftitem("tdls_staffs:air_core", {
  description = "Air Core",
  inventory_image = "air_core.png",
  stack_max = 99
})

minetest.register_craft({
  output = "tdls_staffs:air_core",
  recipe = {
    {"", "ethereal:crystal_spike", ""},
    {"", "default:paper", ""},
    {"", "ethereal:crystal_spike", ""}
  }
})

minetest.register_craft({
  output = "tdls_staffs:staff_air",
  recipe = {
    {"", "tdls_staffs:air_core", ""},
    {"", "default:stick", ""},
    {"", "default:stick", ""}
  }
})
